/*
 * Name: ebcdic.h
 *
 * Function: Provides prototypes and #define for ascii-ebcdic glue layer
 *           routines which support ascii input characters on systems with
 *	     EBCDIC c-libraries.
 */

#ifndef _EBCDIC_H
#define _EBCDIC_H

#include <stdio.h>    		/* need for FILE */
#include <sys/types.h>    	/* need for uid_t, key_t */
#include <dirent.h>   		/* need for DIR */
#include <sys/statvfs.h>  	/* need for statvfs */
#include <sys/stat.h>		/* need for stat */
#include <netdb.h>    		/* need for hostent, servent */
#include <regex.h>    		/* need for regex_t */
#include <utime.h>    		/* need for utimbuf */
#include <nl_types.h> 		/* need for nl_item */
#include <time.h> 		/* need for struct tm */
#include <sys/time.h>		/* need for struct timeval */
#include <spawn.h>		/* need for struct inheritance */
#include <ctype.h>		/* need to get for isxxx before re-defining */
#ifdef IBM_OS390
#include <dynit.h>     		/* need for dynalloc */
#include </usr/include/dll.h>   /* need for dllhandle, 
				   naming conflict with dll.h in mir/mir */
#include <unistd.h>		/* needed to order include files such that blat stuff compiles
				   properly on z/OS */
#endif

#ifdef IBM_AS400
#include <mih/rslvsp.h>
#include <qleawi.h>
#endif /* IBM_AS400 */

#ifdef __cplusplus
extern "C" {
#endif

#ifdef EBCDIC

/* these are hash tables from al_ebcdic.c.  since that file does not include
 * al_ebcdic.h, it is safe to 'extern' these here 
 */
extern char ATOEMAP[];
extern char ETOAMAP[];
extern int ISXXXX[];

/* this is the pointer to the fd table */
extern char * IS_EB_FD;

/* these macros can override the ebcdic translation mode chosen automatically
   by the functions that maintain the file pointer/descriptor */

#define __fp_ebcdic(a,b)  (IS_EB_FD[fileno(a)]=b)
#define __fd_ebcdic(a,b)  (IS_EB_FD[a]=b)

/*************************************************** isxxxxx functions */
/* first, undefine the macros from ctype.h */
#undef isalnum
#undef isalpha
#undef iscntrl
#undef isdigit
#undef isgraph
#undef islower
#undef isprint
#undef ispunct
#undef isspace
#undef isupper
#undef isxdigit
#undef tolower
#undef toupper

/* then, inline the function using the predefined table ISXXXX */
#define isalnum(a)  (ISXXXX[a & 0xFF] & 0x001)
#define isalpha(a)  (ISXXXX[a & 0xFF] & 0x002)
#define iscntrl(a)  (ISXXXX[a & 0xFF] & 0x004)
#define isdigit(a)  (ISXXXX[a & 0xFF] & 0x008)
#define isgraph(a)  (ISXXXX[a & 0xFF] & 0x010)
#define islower(a)  (ISXXXX[a & 0xFF] & 0x020)
#define isprint(a)  (ISXXXX[a & 0xFF] & 0x040)
#define ispunct(a)  (ISXXXX[a & 0xFF] & 0x080)
#define isspace(a)  (ISXXXX[a & 0xFF] & 0x100)
#define isupper(a)  (ISXXXX[a & 0xFF] & 0x200)
#define isxdigit(a) (ISXXXX[a & 0xFF] & 0x400)

/* these don't apply to ISXXXX */
#define tolower(a)  ETOAMAP[tolower(ATOEMAP[a & 0xFF]) & 0xFF]
#define toupper(a)  ETOAMAP[toupper(ATOEMAP[a & 0xFF]) & 0xFF]


/*********************************************************** time  */
#undef ctime
#undef asctime
#undef strftime
#define asctime(a)     __asctime_a(a)
#define ctime(a)       __ctime_a(a)
#define strftime(a,b,c,d)	__strftime_a(a,b,c,d)
#define utime(a,b)     __utime_a(a,b)
#define utimes(a,b)    __utimes_a(a,b)

char *__asctime_a(const struct tm *); 
char *__ctime_a(const time_t *);
size_t __strftime_a (char *, size_t, const char *, const struct tm *);
int __utime_a(const char *, const struct utimbuf *);
#ifndef IBM_OS390 /* don't understand why it conflicts ! */
int __utimes_a(const char *, const struct timeval *);
#endif


/*********************************************************** string parsing */
#define atoi(a)        __atoi_a(a)
#define atof(a)        __atof_a(a)
#define atol(a)        __atol_a(a)
#define strtod(a,b)    __strtod_a(a,b)
#define strtol(a,b,c)  __strtol_a(a,b,c)
#define strtoul(a,b,c) __strtoul_a(a,b,c)
#define getopt(a,b,c)  __getopt_a(a,b,c) 

int __atoi_a(const char *);
double __atof_a(const char *);
long int __atol_a(const char *);
double __strtod_a( const char *, char **);
long int __strtol_a( const char *, char **, int);
unsigned long int __strtoul_a( const char *, char **, int); 
int __getopt_a(int argc, char *const argv[], const char *); 


/*********************************************************** environment */
#define getenv(a)      __getenv_a(a)
#define putenv(a)      __putenv_a(a)
#define setlocale(a,b)   __setlocale_a(a,b)
#define localeconv       __localeconv_a

char *__getenv_a(const char *);
int __putenv_a(const char *);
char *__setlocale_a(int, const char *);                     
struct lconv *__localeconv_a(void); 


/*********************************************************** password */
#define getpass(a)     __getpass_a(a)
#define getpwnam(a)    __getpwnam_a(a)
#define getpwuid(a)    __getpwuid_a(a)
#define getlogin       __getlogin_a

#ifdef IBM_OS390
#define __login		__login_a
#define __passwd	__passwd_a

int __passwd_a(char *username, char *oldpass, char *newpass);
int __login_a(  int function_code, int identity_type, int identity_length, 
		void *identity, int pass_length, char *pass, 
		int certificate_length, char *certificate, int option_flags); 
#endif

char * __getpass_a(const char *);
struct passwd *__getpwnam_a(const char *);
struct passwd *__getpwuid_a(uid_t);  
char *__getlogin_a(void);


/*********************************************************** server/host */
#define getservbyname(a,b) __getservbyname_a(a,b)
#define gethostbyaddr(a,b,c) __gethostbyaddr_a(a,b,c)
#define gethostbyname(a) __gethostbyname_a(a)
#define gethostname(a,b) __gethostname_a(a,b)

struct hostent *__gethostbyaddr_a(char *, int, int);
struct hostent *__gethostbyname_a(char *);
struct servent *__getservbyname_a(char *, char *);
int __gethostname_a(char *, size_t);


/*********************************************************** exec family */
#ifndef IBM_AS400 /* doesn't have these */

#define execv(a,b)     __execv_a(a,b)
#define execve(a,b,c)  __execve_a(a,b,c)
#define execvp(a,b)    __execvp_a(a,b)
#define execl     __execl_a
#define execle    __execle_a
#define execlp    __execlp_a
#define rexec          __rexec_a  

int __rexec_a(char **, int, char *, char *, char *, int *); 
int __execv_a(const char *, char *const []);
int __execve_a(const char *, char *const [], char *const []);
int __execvp_a(const char *, char *const []);

#endif

/*********************************************************** spawn family */
#undef spawn
#undef spawnp
#define spawn(a,b,c,d,e,f) __spawn_a(a,b,c,d,e,f)
#define spawnp(a,b,c,d,e,f) __spawnp_a(a,b,c,d,e,f)
#ifdef IBM_OS390
#define __spawnp2(a,b,c,d,e,f) __spawnp2_a(a,b,c,d,e,f)
#endif
#define popen(a,b)     __popen_a(a,b)

int __spawn_a(const char *, const int, const int [],
              const struct inheritance *, char *const [],
              char *const []);
int __spawnp_a(const char *, const int, const int [],
              const struct inheritance *, char *const [],
              char *const []);
#ifdef IBM_OS390
int __spawnp2_a(const char *, const int, const int [],
              const struct inheritance *, char *const [],
              char *const []);
#endif
FILE *__popen_a(const char *, const char *);


/********************************************************** string compares */
#ifndef IBM_AS400 /* this is in al_proto.h / al_port.c */
#define strcasecmp(a,b)   __strcasecmp_a(a,b)
#define strncasecmp(a,b,c)  __strncasecmp_a(a,b,c)

int  __strcasecmp_a (const char *, const char *);
int  __strncasecmp_a (const char *, const char *, size_t);
#endif


/********************************************************** file control */
#undef getcwd
#undef rename
#define opendir(a)     __opendir_a(a)
#define readdir(a)     __readdir_a(a)
#define chdir(a)       __chdir_a(a)
#define rmdir(a)       __rmdir_a(a)
#define unlink(a)      __unlink_a(a)
#define link(a,b)      __link_a(a,b)
#define creat(a,b)     __creat_a(a,b)
#define access(a,b)    __access_a(a,b)
#define open           __open_a
#define fopen(a,b)     __fopen_a(a,b)
#define fdopen(a,b)    __fdopen_a(a,b)
#define freopen(a,b,c) __freopen_a(a,b,c)
#define remove(a)      __remove_a(a)
#define rename(a,b)    __rename_a(a,b)
#define tmpnam(a)      __tmpnam_a(a)
#define tempnam(a,b)   __tempnam_a(a,b)
#define ptsname(a)     __ptsname_a(a)
#define chown(a, b, c) __chown_a(a, b, c)
#define chmod(a,b)     __chmod_a(a,b)
#define mkdir(a,b)     __mkdir_a(a,b)
#define stat(a,b)      __stat_a(a,b)
#define lstat(a,b)     __lstat_a(a,b)
#define mktemp(a)      __mktemp_a(a)
#define statvfs(a,b)   __statvfs_a(a,b)
#define getcwd(a,b)    __getcwd_a(a,b)
#define readlink(a,b,c) __readlink_a(a,b,c)
#define symlink(a,b)   __symlink_a(a,b)
#ifdef IBM_OS390
#define ttyname(a)     __ttyname_a(a)
#define mknod(a,b,c)   __mknod_a(a,b,c) 
#endif
#if (defined(IBM_AS400) && !(defined(OS4004_5)))
#define mkfifo(a,b)    __mkfifo_a(a,b)
#endif
#define close(a)       __close_a(a)
#ifdef IBM_OS390
#define tmpfile        __tmpfile_a
#define fclose(a)      __fclose_a(a)
#define pclose(a)      __pclose_a(a)
#define dup(a)         __dup_a(a)
#define dup2(a,b)      __dup2_a(a,b)
#define pipe(a)        __pipe_a(a)
#define fcntl          __fcntl_a
#define socket(a,b,c)  __socket_a(a,b,c)
#define accept(a,b,c)  __accept_a(a,b,c)
#endif

int __ioctl_a(int, unsigned long, ...);
#ifdef IBM_OS390
int __fcntl_a(int, int, ...);
int __tcsetattr(int, int, const struct termios *);
int __socket_a(int, int, int); 
int __accept_a(int, struct sockaddr *, size_t *);
FILE * __tmpfile_a(void);
int __fclose_a(FILE *);  
int __pclose_a(FILE *);  
int __dup_a(int);
int __dup2_a(int, int);
int __pipe_a(int x[]); 
#endif
int __close_a(int);
DIR *__opendir_a(const char *); 
struct dirent *__readdir_a(DIR *);
int __chdir_a(const char *);
int   __rmdir_a(const char *);
int   __unlink_a(const char *);
int __link_a(const char *, const char *);
int __creat_a(const char *, mode_t);
int __access_a(const char *, int);
int __open_a(const char *, int, ...);
FILE *__fopen_a(const char *, const char *);
FILE *__fdopen_a(int, const char *); 
FILE *__freopen_a(const char *, const char *, FILE *);
int __remove_a(const char *);
int __rename_a(const char *, const char *);
char  *__tempnam_a(const char *, const char *);
char * __tmpnam_a(char *string);
char *__ptsname_a(int);
int __chown_a(const char *, uid_t, gid_t);
int __chmod_a(const char *, mode_t);
int __mkdir_a(const char *, mode_t);
int __stat_a(const char *, struct stat *);
char *__mktemp_a(char *);
int __statvfs_a(const char *, struct statvfs *);
char *__getcwd_a(char *, size_t);
int __readlink_a(const char *, char *, int);
#ifdef IBM_OS390
char *__ttyname_a(int);
int __mknod_a(const char *, mode_t, rdev_t); 
#endif


/********************************************************** input routines */
#undef getc
#undef getchar
#define fgetc(a)       __getc_a(a)
#define fgets(a,b,c)   __fgets_a(a,b,c)
#define fread(a,b,c,d) __fread_a(a,b,c,d)
#define read(a,b,c)    __read_a(a,b,c)
#define getc(a)        __getc_a(a)
#define gets(a)        __gets_a(a)
#define getchar        __getchar_a
#define scanf          __scanf_a
#define sscanf         __sscanf_a

char *__fgets_a(char *, int, FILE *);
char *__gets_a(char *);
int __fread_a(void *,size_t ,size_t , FILE *);
int __getc_a(FILE *);
int __getchar_a(void);
ssize_t __read_a(int, void *, size_t);
int __scanf_a(const char *, ...);
int __sscanf_a(char *, const char *, ...);

/********************************************************** output routines */
#undef putc
#undef putchar
#define write(a,b,c)    __write_a(a,b,c)
#define printf          __printf_a
#define fprintf         __fprintf_a
#define vprintf         __vprintf_a
#define vfprintf        __vfprintf_a
/* os/390 R6 has a built in ascii sprintf */
#ifndef IBM_OS390
# define sprintf        __sprintf_a
# define vsprintf       __vsprintf_a
#else
# pragma map (__vsprintf_la,"\174\174VSPTF\174")
# pragma map (__sprintf_la,"\174\174SPRTF\174")
# define vsprintf       __vsprintf_la
# define sprintf        __sprintf_la
#endif
#define fputc(a,b)      __putc_a(a,b)
#define fputs(a,b)	__fprintf_a((b), "%s", (a))
#define fwrite(a,b,c,d) __fwrite_a(a,b,c,d)
#define perror(a)       __perror_a(a)
#define ungetc(a,b)	__ungetc_a(a,b)
#define putc(a,b)       __putc_a(a,b)
#define puts(a)		__puts_a(a)
#define putchar(a)      __putchar_a(a)

size_t __write_a(int, const void *, size_t);
int  __printf_a (const char *, ...);
int  __sprintf_a (char *, const char *, ...);
int  __fprintf_a (FILE *, const char *, ...);
int __vfprintf_a(FILE *, const char *, va_list );
int __vsprintf_a(char *, const char *, va_list );
int __fputc_a(int, FILE *);
size_t __fwrite_a(const void *, size_t , size_t , FILE *);
void __perror_a (const char *);

int __ungetc_a(int c, FILE *stream);
int __putc_a(int c, FILE *stream);
int __sprintf_la(char * dest, const char *format, ...);
int __vsprintf_la(char * dest, const char *format, va_list parg);
int __puts_a(char *fstring);

/********************************************************* dll stuff */
#ifdef IBM_OS390
#define dllload(a) __dllload_a(a)
#define dllqueryfn(a,b) __dllqueryfn_a(a,b)

dllhandle * __dllload_a(const char * dllname);
void (* __dllqueryfn_a(dllhandle *dllHandle, char *funcName)) ();
#endif

#ifdef IBM_AS400
#define QleGetExp(a,b,c,d,e,f,g) __QleGetExp_a(a,b,c,d,e,f,g)

void *__QleGetExp_a (int *actmark, int *exp_num, int *exp_len, \
                        char *exp_name, void **symptr, int *exp_type, Qus_EC_t *ec);
#endif

/*********************************************************** miscellaneous */

#define dynalloc(a)    __fynalloc_a(a)
#define getgrnam(a)    __getgrnam_a(a)
#define getwd(a)       __getwd_a(a)
#define uname(a)       __uname_a(a)
#define inet_addr(a)   __inet_addr_a(a)
#define inet_ntoa(a)   __inet_ntoa_a(a)
#define mbstowcs(a,b,c) __mbstowcs_a(a,b,c)
#define mbtowc(a,b,c)  __mbtowc_a(a,b,c)
#define setkey(a)      __setkey_a(a)
#define strerror(a)    __strerror_a(a)
#define system(a)      __system_a(a)
#define nl_langinfo(a) __nl_langinfo_a(a)

#ifndef IBM_AS400 /* has them in al_proto.h / al_port.c */
#undef ftok
#define ecvt(a,b,c,d)  __ecvt_a(a,b,c,d)
#define gcvt(a,b,c)    __gcvt_a(a,b,c)
#define gcvt(a,b,c)    __gcvt_a(a,b,c)
#define ftok(a,b)      __ftok_a(a,b)
#define shmget(a,b,c)  __shmget_a(a,b,c)

char *__ecvt_a(double, int , int *, int *);
char *__fcvt_a(double, int, int *, int *);
char *__gcvt_a(double, int, char *);
key_t __ftok_a(char *, int); 
#else
#define gcvt(a,b,c)    __gcvt_a(a,b,c)
char *__gcvt_a(double, int, char *);
#endif

struct group *  __getgrnam_a(const char *);
char *__getwd_a(char *);

#if defined(IBM_OS390) && !defined(_OE_SOCKETS) && !defined(_OPEN_SOCKETS) && !defined(IBM_AS400)
in_addr_t __inet_addr_a(const char *); 
#else
unsigned long __inet_addr_a(char *); 
#endif
char * __inet_ntoa_a(struct in_addr);

size_t __mbstowcs_a(wchar_t *, const char *, size_t);
int __mbtowc_a(wchar_t *, const char *, size_t);
void __setkey_a(const char *);  
char *__strerror_a(int);
int __system_a( const char * );
extern char *__nl_langinfo_a(nl_item);


/*********************************************************** extra functions */

void __toascii_a(char *, const char *);
void __toebcdic_a(char *, const char *);
int __argvtoascii_a(int, char *[]);
int __argvtoebcdic_a(int, char *[]);

#ifdef IBM_AS400
#define BAANA2E(a, b) BAAN_translate_with_table(a, b, ATOEMAP)     
#define BAANE2A(a, b) BAAN_translate_with_table(a, b, ETOAMAP)
void BAAN_translate_with_table(char *, size_t, char *);
void glue_init( void );
int set_jobccsid_to_500( void );
int retrieve_job_ccsid( void );
int retrieve_system_ccsid( void );
int sli_init_glue(void(*)(int*, int*));
#endif

/*********************************************************** assert macro */
/* The assert function produce unreadable messages because of ascii-ebcdic
   mismatches. Therefore we have to remap it on AS/400 for our own macro.
   This macro is based on the system assert() macro defined in assert.h.*/
#ifdef IBM_AS400
# undef assert

# define MAX_ASSERT_MSG_LEN	1024
# define MAX_SRC_FILE_NAME_LEN	 500
# define MAX_EXPR_LEN		 500
# define assert(expr) { char assertStr[MAX_ASSERT_MSG_LEN], file[MAX_SRC_FILE_NAME_LEN], _expr[MAX_EXPR_LEN];\
                           if( (expr) )                                                 \
                                ((void)0);                                              \
                           else {                                                       \
                                __toebcdic_a(file, __FILE__);   \
                                __toebcdic_a(_expr, #expr);             \
                                sprintf(assertStr, "%s", _C_assrt(_expr, file, __LINE__));      \
                                BAANE2A(assertStr, strlen(assertStr));                  \
                                fprintf(stderr,"%s \n",assertStr);                              \
                                system("DSPJOB OUTPUT(*PRINT) OPTION(*ALL)");           \
                                abort();                                                \
                           } };
#endif

#endif /* EBCDIC */

#ifdef __cplusplus
}
#endif

#endif /* _EBCDIC_H */

