#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bdb_api.h"

#include "products.h"

BDB_ADR_LINK Product_Link[] = {
	{ "ttexa100.code"    , (char *) &Product.Code    },
	{ "ttexa100.descr"   , (char *)  Product.Descr   },
	{ "ttexa100.price"   , (char *) &Product.Price   },
	{ "ttexa100.Refcntd" , (char *) &Product.Refcntd },
	{ "ttexa100.Refcntu" , (char *) &Product.Refcntu },
	{ "ttexa100._compnr" , (char *) &Product._compnr },
	{ "ttexa100._dlock"  , (char *) &Product._dlock  },
	{ ""                 , (char *) 0                }};


/*********************************************************************
 This text contains functions that concern customization of the
 SQL-library and a few support functions to simplify the use of
 functions within the BDB- and SQL-library.
*/

static	char	dummy_sym;

/*********************************************************************
 These variables are global to avoid elaborate registrations in the
 demo. Note that in more realistic approaches more cursors and
 corresponding datadictionary tables occur in an application.
 Custodial registrations for these resources are needed than.
*/
BDB_MAIN_CURSOR *cursor;           /* main cursor */
DD_TABLE	*dd_table;     /* data dictionary table */
SqlVal		 limit;


/*********************************************************************
 Error handler function for TSS 
*/
static void	tss_error_handler(int type, char *str)
{
	switch (type) {
		case 1: /*FATAL*/
			printf("TSS Fatal error: %s\n", str);
			break;
		case 2: /*WARNING*/
			printf("TSS Warning: %s\n", str);
			break;
	}
}

/*********************************************************************
 This function abstracts from a few introductive functions that are
 needed in the BDB- as well as in the SQL-library.
*/
void init_dbc (char *progname, int progmode, char *userid)
{
	bdb_init_api(progname, progmode, 0, tss_error_handler); /* last-1 argument = 1 when shared memory should be used */

	/* Connect to BDB-library functionalities */
	bdb_logon (userid);
}

/*********************************************************************
 This function abstracts from a few initialisations in the SQL-library.
 It determines debug mode and redirects some functionvectors used
 as up-calls from the SQL-library.
*/
/* Due to forward declarations... */
static int my_sql_for_update(void *table_info, UCHAR *row_buffer, void **not_used, int re_lock);

static int my_qpc_error(int line, char *errmsg);

static char *my_qpc_crt_extvar (char *name, BDB_TYPE *type, void **table_info, int *dims, ULONG mal_id);

static int my_qpc_put_extvar (void *csym, void *addr, BDB_TYPE type, int size, int *dims);

static int my_qpc_get_extvar (void *csym, SqlVal *value, BDB_TYPE *type, int *size);


void customize_sql()
{
   init_qpc_debug ();
   sql_init_func (FUNC_SQL_ERROR,(SqlFunc)my_qpc_error);
   sql_init_func (FUNC_SQL_CRT_EXTVAR,(SqlFunc)my_qpc_crt_extvar);
   sql_init_func (FUNC_SQL_PUT_EXTVAR,(SqlFunc)my_qpc_put_extvar);
   sql_init_func (FUNC_SQL_GET_EXTVAR,(SqlFunc)my_qpc_get_extvar);
   sql_init_func (FUNC_SQL_UPDATE,(SqlFunc)my_sql_for_update);
}


/*********************************************************************
 Customize error output.
 This up-call can be constructed to fit into your own user-interface.
*/
static int my_qpc_error (int line, char *errmsg)
{
   return printf ("Error in line %d : %s\n",line,errmsg);
}

/*********************************************************************
 Customize output of data to the application area.
*/
static int my_qpc_put_extvar (void *csym, void *addr, BDB_TYPE type, int size, int *dims)
{
   if ( (!csym) || (csym == &dummy_sym) )
	return (0);

   switch (type) {
      /* type is BDB_CHAR, BDB_LONG, etc. Is not used since always a
         memcpy can be done into the user_addres, irrespective of the type.
      */
   case BDB_LONG:
      *(long *)(((BDB_U_ABIND *)csym)->user_adres) = bs4_out(addr);
      break;

   default:
      memcpy (((BDB_U_ABIND *)csym)->user_adres,addr,size);
      break;
   }
   return (0);
}
/*********************************************************************
 Customize input of data from the application area.
*/
static int my_qpc_get_extvar (void *csym, SqlVal *value, BDB_TYPE *type, int *size)
{
   /* normally, you read the external variable, convert it to long, double or a string
      and put the converted value in SqlVal * value */

   value->d = limit.d;
   *type    = BDB_DOUBLE;
   *size    = sizeof(double);
   return (0);
}

/*********************************************************************
 Customize creation of application area.
*/
static char *my_qpc_crt_extvar (char *name, BDB_TYPE *type, void **table_info, int *dims, ULONG mal_id)
{
   /* because in this demo application, there is only 1 cursor, always
      this cursor can be set to table_info. In case more tables are used,
      the cursor corresponding to column/table's name has to be selected
      and assigned to table_info.

      This function must always return 'something'. If no specific symbol
      has to be returned, then return '&dummy_sym'. In the function
      my_sql_get_extvar() this value is checked again.
   */
   int idx;

   switch (*type) {
	   case 1: /* column name */
   		   idx = bind_idx(name);
		   if (table_info)
		      *table_info = cursor;
		   return ((char * )&Product_Bind[idx]);
	   case 2: /* table name */
		   if (table_info)
		      *table_info = cursor;
		   return (&dummy_sym);
	   case 3: /* other than table/column name */
		   /* not used in this demo application */
		   return (&dummy_sym);
   }
   return (&dummy_sym);
}

/*********************************************************************
 Customize function that handles what to do in locking situations.
*/
static int my_sql_for_update(void *table_info, UCHAR *row_buffer, void **not_used, int re_lock)
{
   BDB_MAIN_CURSOR *main_cursor = (BDB_MAIN_CURSOR*)table_info; 
		/* table_info is a cursor type in this example */
   int   lock_value;             
   int   ret;                    
   
   BDB_CURSOR	*internal_cursor;           /* internal cursor */
   UCHAR	record[MAX_ROW_SIZE];       /* will work for all tables */

   /* not_used is NOT used by the query processor and will always be NULL */

   if (bdb_determine_row_cursor(&internal_cursor, main_cursor) < 0)
      return (-1);

   if (re_lock) {
      if (init_dlock_offset(internal_cursor, re_lock))
         return (-1);
   }

   lock_value = put_dlock_row(internal_cursor, row_buffer);
   printf ("lock_value = %d\n",lock_value);
   if (lock_value >= 0) {
      bdb_bind_to_buf(main_cursor, record);
      /* dd_table is a global variable, in real life applications
	 you will get this info though table_info */
      put_dlock_in_row(dd_table, record, lock_value);
      ret = bdb_buf_to_bind(main_cursor, record);
      return (lock_value);
   }
   return (-1);
}

/*********************************************************************
 Support function to read an external datadictionary description into
 a BDB_U_ABIND structure.
 This function might become part of the BDB-library one time.
*/
BDB_U_ABIND *dd2bind(char *tname, BDB_ADR_LINK *user_link, int *nr_attr)
{
   DD_TABLE	*dd;
   DD_COLUMN	*ddcol;
   BDB_U_ABIND	*walk_bind, *halt_bind;
   BDB_ADR_LINK	*walk_link;
   int		loop;

   dd = ddtable(tname);

   if (dd == NULL) {
	printf("errno = %d\n", bdb_errno);
	exit(1);
   }

   /* Determine space for bind array and allocate it */
   ddcol = dd->columns;
   loop = dd->no_dd_columns;
   *nr_attr = loop;
   while (--loop >= 0) {
      /* skip combined fields */
      if (ddcol->type == BDB_COMBINED) (*nr_attr)--; 
      ddcol++;
   }
   halt_bind = (BDB_U_ABIND *) malloc (*nr_attr * sizeof (BDB_U_ABIND));

   /* Load bind array from datadictionary, user address excepted */
   ddcol = dd->columns;
   loop = dd->no_dd_columns;
   walk_bind = halt_bind;
   while (--loop >= 0) {
      /* skip combined fields */
      if (ddcol->type == BDB_COMBINED) {
	 ddcol++;
	 continue;
      }
      walk_bind->user_adres = (char *) 0;
      walk_bind->size_entry = ddcol->size / ddcol->dept;
      walk_bind->no_entry   = ddcol->dept;
      walk_bind->ext_type   = ddcol->type;
      walk_bind->diga       = ddcol->detail.domain->diga;
      walk_bind->rndo       = ddcol->detail.domain->rndo;
      strcpy(walk_bind->ext_name, ddcol->name);
      walk_bind++;
      ddcol++;
   }

   /* Complete bind array with user addresses from table user_link */
   walk_link = user_link;
   /* For all members in user_link */
   while (strcmp(walk_link->column,"")) {
      int found = FALSE;
      walk_bind = halt_bind;
      loop = *nr_attr;
      /* Locate matching column name in bind table */
      while ((--loop>=0) & (!found)) {
  	  found = (strcmp (walk_link->column,walk_bind->ext_name)==0);
	  walk_bind++;
      }
      if (found) {
	  walk_bind--;
	  walk_bind->user_adres = walk_link->user_address;
      } else {
	  printf("Column \"%s\" not found in DD.\n",walk_link->column);
      }
      walk_link++;
   }
   return (halt_bind);
}

/*********************************************************************
 Support function to convert columnnames to indexes that refer to
 elements in a BDB_U_ABIND array, complying to creation with dd2bind().
*/
int bind_idx (char *column_name)
{
   DD_TABLE	*dd;
   DD_COLUMN	*ddcol;
   int          found;
   int          attr_nr;
   int		loop;
   char         *match;
   char         table_name[COLUMN_NAME_LEN + 1]; /* based on column... */

   strcpy (table_name,column_name); /* ...because of this ! */
   match = strchr (table_name,'.');
   if (match) {
      *match = '\0';            /* Replace dot with terminator */
      dd = ddtable(table_name);

      /* Locate column while counting valid attributes */
      ddcol = dd->columns;
      attr_nr = 0;
      loop = dd->no_dd_columns;
      found = FALSE;
      while ((--loop>=0) & !found) {
         if (ddcol->type != BDB_COMBINED) {
	     attr_nr++; /* count valid attributes */
	     found = (strcmp(column_name,ddcol->name)==0);
	 }
         ddcol++;
      }

      if (found) {
	  attr_nr--;
	  return (attr_nr);
      } else {
	  return (-1); /* column_name not found */
      }
   } else {
     return (-1); /* illegal column_name */
   }
}
/*********************************************************************/
