/*
 * ddc_server.c
 */

#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>


#ifdef __WIN32__
# include <windows.h>
# include <process.h>
# include <io.h>
#else	/* __WIN32__ */
# include <sys/msg.h>
# include <sys/ipc.h>
#endif

#include "system.h"
#include "ddc_baan.h"
#include "ddc.h"
#include "ddc_supplier.h"
#include "ddc_tools.h"

#define LOGGING_OFF	0
#define LOGGING_ON	1

#define TESTLVFS	0
#define MAXLISTS	10
#define MAXLISTLEN	15
#define MAXLISTVAL	99998

#define SERVER_SENT 	1
#define SERVER_REC 	2

#define REQU_NUM_PARMS	14

#define TEST_UNEXPECTED_EXIT	0  /* This is used to test some unusual cases. */
				   /* If this is set to zero, the server runs  */
				   /* and exits normally.  If set to one, the  */
				   /* server requests an exit before the driver*/
				   /* is finished.  If set to two, the server  */
				   /* removes the message queues and quits     */
				   /* without sending the proper ending        */
				   /* messages to the driver.                  */

#define TEST_BAD_REPLIES	1  /* This is used to allow the server to send */
				   /* bad reply messages to the driver for     */
				   /* testing.  If this is set to one, bad     */
				   /* reply messages will be occasionally      */
				   /* generated and sent to the driver.        */


typedef struct _FUNC_FILE
{
	char	*sname,
		*fname,
		*trans_pre;
	FILE	*fd;
	struct _FUNC_FILE	*next;
} FUNC_FILE;


/* Globals			*/
#ifdef __WIN32__
__declspec (dllimport) FILE             *debug_fd;
#else
extern FILE	*debug_fd;
#endif
FILE		*menufd, *infd;
int		ddc_debug = LOGGING_OFF;
int		LVF_file_empty = 0;
int		menu_file_open = 0;
GROUPLIST	*groups, *selected;
FUNC_FILE	*fservers;
static char	sep[2];

int             ddc_initiate;
static int	msgid_in;
static int	msgid_out;
static int	msgid_LVF;
static MSGFORM	msgin;
static MSGFORM	msgout;
static MSGFORM	msgLVF;


/* Local procedures		*/
SignalType	clean_kill (int);
int		generate_LVF_file (FILE *);
int		send_LVF (FILE *);
static	int    	svr_write_mesg (int, char *);
static	int    	svr_get_field (int, char *, char *);
int		in_group_list (char *);
int		get_group_list ( void );
int		free_group_list ( void );
int		send_selected_groups ( void );
int		in_function_server_list (FUNC_FILE *);
int		add_function_server (char *, char *);
int		send_menu_msg (char *, char *);
int		send_menus ( void );
int		remove_function_server_list ( void );
int		get_input_fd (FUNC_FILE *);
int		get_reply_message (char *, FUNC_FILE, char *);
int		get_bad_reply ( long*, char* );

SignalType	clean_kill( input )
int input;
{
	ddc_stop_server();
	ddc_ready_to_stop();
	exit(0);
}


int	generate_LVF_file(fd)
FILE	*fd;
/*
	IN:   fd	The descriptor of the file to store the LVF lists in.

	OUT:		The file "LVFfile" is created and a list of LVF messages
			are written to it.
	      Returns:	0, if the lists are placed in the file without problems.
			-1, if an error occurs.

	PURPOSE:  This function generates the LVF lists to be used in testing the 
		  universal driver.  First, for each LVF list generated, a random
		  number between one and MAXLISTLEN is generated to determine how
		  many elements will be placed in the list.  Once the number of
		  elements for a list is determined, the list elements are
		  randomly generated and a list start, list end, and the list
		  elements are placed in the file.  When all of the lists are
		  generated, the function returns.
*/
{
int	a, b, namelen, elements, list[MAXLISTLEN];
char	name[MAXNAMELEN];

	fd = fopen("LVFfile", "w");

	for(a = 0; a < MAXLISTS; a++)
	{
		namelen = (wrap_rnd_ri(MAXNAMELEN) + 1);

		for(b = 0; b < namelen; b++)
			name[b] = get_letter(wrap_rnd_ri(26));

		name[b] = '\0';

		elements = (wrap_rnd_ri(MAXLISTLEN) + 1);

		for(b = 0; b < elements; b++)
			list[b] = (wrap_rnd_ri(MAXLISTVAL) + 1);

		/* Do sort here if wanted. */

		/* Place LVF data in file, start with DDC_LIST_START message. */
		fprintf(fd, "ddc_driver%sDDC_LIST_START%s%s%s%d%s1%s\n",
			sep, sep, name, sep, a + 1, sep, sep);

		/* Add DDC_LIST_ELEM message for each element of the LVF list. */
		for(b = 0; b < elements; b++)
			fprintf(fd, "ddc_driver%sDDC_LIST_ELEM%s%s%sA%s%d\n",
				sep, sep, name, sep, sep, list[b]);

		/* Close the LVF list with a DDC_LIST_END message. */
		fprintf(fd, "ddc_driver%sDDC_LIST_END%s%s%s%d%s0%s0%s%d\n",
			sep, sep, name, sep, elements, sep, sep, sep, a + 1);

		memset(name, 0, MAXNAMELEN);
	}

	fprintf(fd, "\n\n");

	fclose(fd);

	return 0;

} /* end generate_LVF_file() */


static	int    svr_get_field( fnum, msgtest, field )
int     fnum;
char    *msgtest,
	*field;
/*
	IN:   fnum	An integer value that determines which field to get.
	      msgtext	A string containing the message to get a field from.
	      field	A string to store the value of the desired field in.

	OUT:  Returns:	The value of the desired field in field.
			len, The length of the field's value if successful.
			-1, if an error occured.

	PURPOSE:  This function takes an integer value, a message string, and a
		  buffer to store the desired output in.  It searches the message
		  string until it finds the field which corresponds to the given
		  integer value.  Once found, it copies the contents of the
		  desired field into the buffer and returns the string length of
		  the buffer.  If there is not a field which corresponds to the
		  given value, -1 is returned.
*/
{
	register  int	len = 0;
	int     	j = 1;
	char		*p1 = msgtest, *p2, sepval;

	sepval = ',';

	if ( p1 && field ) {
		while( j < fnum && *p1 != '\0' ) {
			if ( *p1 == sepval )
				j++;
			p1++;
		}
		if ( *p1 == '\0' )
			return( -1 );

		p2 = field;
		while( *p1 != sepval && *p1 != '\0' )
		{
			*p2++ = *p1++;
			len++;
		}
		*p2 = '\0';

		return( len );
	}

	return( -1 );	/* Otherwise */
}


static int	srv_write_mesg( who, out )
int	who; 
char	*out;
/*
	IN:   who	An integer value which determines which prefix to attach.
	      out	A string containing the message to be written.

	OUT:  Returns:	The message is created and written to the logging file.
			0, if there were no problems.
			-1, if an error occured.

	PURPOSE:  This function writes a message to the logging file, if logging
		  has been turned on.  If logging has been turned on, the
		  function first checks the value of who and writes a prefix to
		  the log file based on the value.  Then, it writes the message
		  given in out to the file and returns zero.  If logging has not
		  been turned on, the function simply returns zero.
*/
{
char	*x;
	
	if (ddc_debug == LOGGING_ON)
	{
		x = (char *) calloc (1, 20);
		if (who == SERVER_SENT)
		{
			strcopy(x, "Server sent: ");
			fwrite(x, strlen(x)+1, 1, debug_fd);
		}
		else
		{
			strcopy(x, "Server received: ");
			fwrite(x, strlen(x)+1, 1, debug_fd);
		}
		fwrite(out, strlen(out)+1, 1, debug_fd);
		strcopy(x, "\n");
		fwrite(x, strlen(x)+1, 1, debug_fd);
		fflush(debug_fd);
		free(x);
	}

	return 0;
}


int	send_LVF(fd)
FILE	*fd;
/*
	IN:   fd	The file descriptor of the file that contains the
			LVF list to be sent.

	OUT:  Returns:	0, if the LVF file was read and the list was sent.
			-1, if an error occured.

	PURPOSE:  This function sends an LVF list to the driver.  It reads the
		  list from a file that contains the messages to send up to
		  MAXLISTS LVf lists.
*/
{
	char		*msg;
	struct tm	*now;

	msg = (char *) calloc (1, sizeof(char) * MAXMSGLEN);

	/* Get the current time and store it in msg in YYYYMMDDHHMMSS format. */
	get_time(&now);

	sprintf(msg, "%04d%02d%02d%02d%02d%02d",
		now->tm_year + 1900, now->tm_mon, now->tm_mday,
		now->tm_hour, now->tm_min, now->tm_sec );

	/* Send the DDC_LIST_START message.  Attach the current time first. */
	memset(msgout.mtext, 0, MAXMSGLEN);
	fgets(msgout.mtext, MAXMSGLEN, fd);

	/* If the LVF file is empty, set the LVF_file_empty var and return. */
	if (!msgout.mtext)
	{
		LVF_file_empty = 1;
		free( msg );
		return 0;
	}

	strgcat(msgout.mtext, msg);

	msgout.mtype = DDC_LIST_START;

	msgout.mtype = DDC_LIST_ELEM;
	srv_msgsnd(msgid_out, &msgout, strlen(msgout.mtext)+1, 0);
	srv_write_mesg(SERVER_SENT, msgout.mtext);

	/* Send the DDC_LIST_ELEM messages. */
	memset(msg, 0, MAXMSGLEN);
	fgets(msgout.mtext, MAXMSGLEN, fd);
	svr_get_field(2, msgout.mtext, msg);

	while(!strcmp(msg, "DDC_LIST_ELEM"))
	{
		msgout.mtype = DDC_LIST_ELEM;
		srv_msgsnd(msgid_out, &msgout, strlen(msgout.mtext)+1, 0);
		srv_write_mesg(SERVER_SENT, msgout.mtext);

		memset(msg, 0, MAXMSGLEN);
		fgets(msgout.mtext, MAXMSGLEN, fd);
		svr_get_field(2, msgout.mtext, msg);
	}

	/* Send the DDC_LIST_END message. */
	msgout.mtype = DDC_LIST_END;
	srv_msgsnd(msgid_out, &msgout, strlen(msgout.mtext)+1, 0);
	srv_write_mesg(SERVER_SENT, msgout.mtext);

	free(msg);

	return 0;

} /* end send_LVF() */


int	in_group_list(gid)
char	*gid;
/*
	IN:   gid	The group id to be checked.
	OUT:  Returns:	0, if the group is in the group list.
			-1, if the group is not in the group list.

	PURPOSE:  This function takes a group id and checks whether or
		  not the function is in the group list that was recieved
		  from the server.
*/
{
GROUPLIST	*temp;

	temp = groups;

	while(temp)
	{
		if (!strcmp(gid, temp->grpident))
			return 0;

		temp = temp->lnext;
	}

	return -1;

} /* end in_group_list() */


int	get_group_list()
/*
	IN:   none
	OUT:  Returns:	0, if the group list was received correctly.
			-1, if an error occured.
	PURPOSE:  This function receives the list of posible groups from
		  the driver.  As each new element is received, it is added
		  to the list of valid groups for later reference.
*/
{
GROUPLIST	*temp;

	groups = NULL;

	do {
		fprintf( stderr, "Server: WAITING for a MESSAGE\n" );
		fprintf( stderr, "Server: WAITING for a MESSAGE\n" );
		srv_msgrcv( msgid_in, &msgin, MAXMSGLEN, 0, DDC_WAIT ); 
		fprintf( stderr, "Server: Msgtype: %ld\n", msgin.mtype );
		srv_write_mesg(SERVER_REC, msgin.mtext);

		switch( msgin.mtype ) {
		case DDC_LIST_ELEM: {
			fprintf( stderr, "A listelem: '%s'\n", msgin.mtext );

			temp = (GROUPLIST *) calloc(1, sizeof(GROUPLIST));
			svr_get_field(ELEM_LE, msgin.mtext, temp->grpident);

			if (!in_group_list(temp->grpident))
			{
				free(temp);
				break;
			}

			temp->lnext = groups;
			groups = temp;

			break;
			}
		case DDC_LIST_END:
			{
			fprintf( stderr, "A listend: '%s'\n", msgin.mtext );
			break;
			}
		default: {
			fprintf( stderr, "Default: '%s'\n", msgin.mtext );
			break;
			}
		}
	} while ( msgin.mtype != DDC_LIST_END );

	return 0;

} /* end get_group_list() */


int	free_group_list( void )
/*
	IN:   None

	OUT:  Returns:	0, if the group list is freed correctly.
			-1, if an error occured.

	PURPOSE:  This function simply frees the memory that is being used
		  to store a group list.  It loops through each element of
		  the list, freeing the memory for the element, until all
		  of the elements have been freed.
*/
{
GROUPLIST	*temp;

	temp = groups;

	while(groups != NULL)
	{
		temp = groups;
		groups = temp->lnext;

		memset(temp->grpident, 0, _GRP_ID);
		temp->lnext = NULL;

		free(temp);
	}

	return 0;

} /* end free_group_list() */


int	send_selected_groups()
/*
	IN:   none
	OUT:  Returns:	0, if the selected groups were send correctly.
			-1, if no groups were selected.

	PURPOSE:  This function sends the list of groups to be selected
		  for use to the driver.  This function opens the file
		  given by the environment variable MENU_FILE and reads
		  the list of groups to select from it, one at a time,
		  until a blank line is read.  For each group in the list,
		  in_group_list() is called to make sure that the group
		  is in the group list.  If it is, the group is added to
		  the selected group list and a mesage is sent to the
		  driver.  If it is not, the loop continues with the next
		  line of the input file.
*/
{
char	*envstr, str[MAXMSGLEN];
int	none_selected = 1;
GROUPLIST	*temp;

	selected = NULL;
	memset(str, 0, MAXMSGLEN);

	envstr = getvar("MENU_FILE");

	if (!envstr)
	{
		menu_file_open = -1;
		fprintf(stderr, "Error: Environment Variable MENU_FILE not set \n");
		return menu_file_open;
	}

	menufd = fopen(envstr, "r");
	menu_file_open = 0;

	envstr = (char *) calloc (1, sizeof(char) * MAXMSGLEN);

	memset(envstr, 0, MAXMSGLEN);
	fgets(envstr, MAXMSGLEN, menufd);

	strncpy(str, envstr, strlen(envstr) - 1);
	strgcat(str, "\0");

	while(strlen(str) != 0)
	{
		if (in_group_list(str))
		{
			memset(envstr, 0, MAXMSGLEN);
			memset(str, 0, MAXMSGLEN);
			fgets(envstr, MAXMSGLEN, menufd);
			continue;
		}

		none_selected = 0;

		temp = (GROUPLIST *) calloc(1, sizeof(GROUPLIST));

		strcopy(temp->grpident, str);
		temp->lnext = selected;
		selected = temp;

		msgout.mtype = DDC_LIST_ELEM;

		sprintf( msgout.mtext, "%s%s%s%s%s%s%s%s%s",
			"ddc_server", (char *) sep, "DDC_LIST_ELEM",
			(char *) sep, "grp_sel", (char *) sep, "A",
			(char *) sep, str);

		srv_msgsnd(msgid_out, &msgout, strlen(msgout.mtext)+1, 0);

		memset(envstr, 0, MAXMSGLEN);
		memset(str, 0, MAXMSGLEN);

		fgets(envstr, MAXMSGLEN, menufd);

		strncpy(str, envstr, strlen(envstr) - 1);
		strgcat(str, "\0");
	}

	free(envstr);

	free_group_list( );
	groups = selected;

	if (none_selected)
		return -1;

	return menu_file_open;

} /* end send_selected_groups() */


int	in_function_server_list(item)
FUNC_FILE	*item;
/*
	IN:   item	The function server to be checked.

	OUT:  Returns:	0, if the function server is in the function
			   server list.
			-1, if the function server is not in the function
			   server list.

	PURPOSE:  This function simply determines whether or not the given
		  function server is in the current function server list.
*/
{
FUNC_FILE	*temp;

	temp = fservers;

	while(temp != NULL)
	{
		if ((!strcmp(item->sname, temp->sname)) &&
		   (!strcmp(item->fname, temp->fname)) &&
		   (!strcmp(item->trans_pre, temp->trans_pre)))
			return 0;

		temp = temp->next;
	}

	return -1;

} /* end in_function_server_list() */


int	add_function_server(str, field)
char	*str, *field;
/*
	IN:   str	A string containing information about the function
			to be added.
	      field	The name of the input file to be associated with
			the new function;

	OUT:  Returns:	0, if the function was added to the function list.
			-1, if an error occured.

	PURPOSE:  This function adds a function server to the list of current
		  function servers.  First, it checks to make sure that the
		  given server doesn't already exist.  If it does, an error
		  is returned.  If it doesn't, a new node is added to the
		  function server list and its information is filled in using
		  the given input.
*/
{
FUNC_FILE	*item;
char		*infile;

	if (strlen(field) > MAXNAMELEN)
		return -1;

	infile = (char *) calloc(1, sizeof(char) * MAXNAMELEN);

	item = (FUNC_FILE *) calloc(1, sizeof(FUNC_FILE));
	item->sname = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
	item->fname = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
	item->trans_pre = (char *) calloc(1, sizeof(char) * MAXNAMELEN);

	svr_get_field(2, str, item->fname);
	svr_get_field(3, str, item->sname);
	svr_get_field(4, str, item->trans_pre);

	strncpy(infile, field, strlen(field) - 1);
	strgcat(infile, "\0");

	item->fd = fopen(infile, "r");

	if (!in_function_server_list(item))
	{
		memset(item->sname, 0, MAXNAMELEN);
		memset(item->fname, 0, MAXNAMELEN);
		memset(item->trans_pre, 0, MAXNAMELEN);

		free(item->sname);
		free(item->fname);
		free(item->trans_pre);
		free(item);

		return -1;
	}

	item->next = fservers;
	fservers = item;

	return 0;

} /* end add_function_server() */


int	send_menu_msg(gid, str)
char	*gid, *str;
/*
	IN:   gid	A string containing the group id of the menu message
			to be sent.
	      str	The rest of the message to be sent;

	OUT:  Returns:	0, if the message was sent correctly.
			-1, if an error occured.

	PURPOSE:  This function sends a menu structure message to the driver.
*/
{
char	type[MAXMSGLEN];

	memset(type, 0, MAXMSGLEN);

	sprintf(msgout.mtext, "%s%s%s%s", gid, sep, str, sep);

	svr_get_field(1, str, type);

	if (!strcmp(type, "DDC_NUMBER"))
		msgout.mtype = DDC_NUMBER;
	else if (!strcmp(type, "DDC_MENU"))
		msgout.mtype =  DDC_MENU;
	else if (!strcmp(type, "DDC_FUNCTION"))
		msgout.mtype = DDC_FUNCTION;
	else if (!strcmp(type, "DDC_COMBINED_FIELD"))
		msgout.mtype = DDC_COMBINED_FIELD;
	else
		return -1;

	srv_msgsnd(msgid_out, &msgout, strlen(msgout.mtext) + 1, 0);
	srv_write_mesg(SERVER_SENT, msgout.mtext);

	return 0;

} /* end send_menu_msg() */


int	send_menus()
/*
	IN:   none

	OUT:  Returns:	0, if the menu structures for the selected groups
			   were sent properly.
			-1, if no menu structures were sent.

	PURPOSE:  This function reads menu structures from the input file
		  that was used to get the list of selected groups and sends
		  them to the driver.  Each time that a new menu structure
		  is read, the function checks to make sure that the group
		  is in the list of selected groups.  If it is, the menu
		  structure is sent.  If it isn't, the menu structure is
		  skipped, and the next one is read.  This function returns
		  zero if the file didn't contain any menu structures or none
		  of the menu structure that the file contained were for
		  selected groups.
*/
{
char	*str, *gid, *buff, *field, *temp;
int	none_sent = 1;

	if (menu_file_open)
		return menu_file_open;

	str = (char *) calloc(1, sizeof(char) * MAXMSGLEN);
	gid = (char *) calloc(1, sizeof(char) * MAXMSGLEN);
	buff = (char *) calloc(1, sizeof(char) * MAXMSGLEN);
	field = (char *) calloc(1, sizeof(char) * MAXMSGLEN);
	temp = (char *) calloc(1, sizeof(char) * MAXMSGLEN);

	memset(str, 0, MAXMSGLEN);
	memset(gid, 0, MAXMSGLEN);
	memset(buff, 0, MAXMSGLEN);
	memset(field, 0, MAXMSGLEN);

	fservers = NULL;

	fgets(str, MAXMSGLEN, menufd);

	strncpy(gid, str, strlen(str) - 1);
	strgcat(gid, "\0");

	while(strlen(gid) > 0)
	{
		if (in_group_list(gid))
		{
			while(strlen(gid) > 0)
			{
				fgets(str, MAXMSGLEN, menufd);

				strncpy(gid, str, strlen(str) - 1);
				strgcat(gid, "\0");
			}

			memset(str, 0, MAXMSGLEN);
			memset(gid, 0, MAXMSGLEN);

			fgets(str, MAXMSGLEN, menufd);

			strncpy(gid, str, strlen(str) - 1);
			strgcat(gid, "\0");

			continue;
		}

		none_sent = 0;

		memset(str, 0, MAXMSGLEN);
		memset(buff, 0, MAXMSGLEN);

		fgets(str, MAXMSGLEN, menufd);

		strncpy(buff, str, strlen(str) - 1);
		strgcat(buff, "\0");

		while(strlen(buff) > 0)
		{
			svr_get_field(1, str, field);

			if (!strcmp(field, "DDC_FUNCTION"))
			{
				while(!strcmp(field, "DDC_FUNCTION"))
				{
					send_menu_msg(gid, buff);

					strcopy(temp, buff);

					memset(str, 0, MAXMSGLEN);
					memset(buff, 0, MAXMSGLEN);

					fgets(str, MAXMSGLEN, menufd);

					strncpy(buff, str, strlen(str) - 1);
					strgcat(buff, "\0");

					svr_get_field(1, str, field);
				}

				add_function_server(temp, field);
			}

			send_menu_msg(gid, buff);

			memset(str, 0, MAXMSGLEN);
			memset(buff, 0, MAXMSGLEN);

			fgets(str, MAXMSGLEN, menufd);

			strncpy(buff, str, strlen(str) - 1);
			strgcat(buff, "\0");
		}

		memset(str, 0, MAXMSGLEN);
		memset(gid, 0, MAXMSGLEN);

		fgets(str, MAXMSGLEN, menufd);

		strncpy(gid, str, strlen(str) - 1);
		strgcat(gid, "\0");
	}

	free(str);
	free(gid);
	free(buff);
	free(field);

	if (none_sent)
		return -1;

	return 0;

} /* end send_menus() */


int	remove_function_server_list()
/*
	IN:   None

	OUT:  Returns:	0, if the function server list is removed properly.
			-1, if an error occured.

	PURPOSE:  This function removes the function server list, closes the
		  input file associated with each of the function servers, and
		  frees the memory that is being used by the list.
*/
{
FUNC_FILE	*temp;

	temp = fservers;

	while(fservers != NULL)
	{
		fservers = temp->next;

	/*	fclose(temp->fd); */
		free(temp->sname);
		free(temp->fname);
		free(temp->trans_pre);
		free(temp);

		temp = fservers;
	}

	return 0;

} /* end remove_function_server_list() */


int	get_input_fd(server)
FUNC_FILE	*server;
/*
	IN:   server	A pointer to the function server that we want to get
			the input file fd for.

	OUT:  Returns:	The fd of the input file associated with the given
			server in server.fd, if the given function server is
			found.
			0, if the given function server is found.
			-1, if the given function server is not found.

	PURPOSE:  This procedure checks to make sure that the given function
		  server exists.  If it does, the fd of the input file
		  corresponding to the given function server is stored in
		  server.fd and 0 is returned.  If it doesn't exist, -1 is
		  returned.
*/
{
FUNC_FILE	*temp;

	temp = fservers;

	while(temp)
	{
		/* If the given server name and function name are equal to those
		   of the current one in the list or there is a non-null
		   transaction prefix which is equal to that of the current
		   function server in the list, we have a match.  Otherwise,
		   continue searching.       */
		if (((!strcmp((*server).sname, temp->sname)) &&
		   (!strcmp((*server).fname, temp->fname))) ||
		   ((!strcmp((*server).trans_pre, temp->trans_pre)) &&
		   (strlen((*server).trans_pre))))
		{
			(*server).fd = temp->fd;
			return 0;
		}

		temp = temp->next;
	}
	
	(*server).fd = NULL;

	return -1;

} /* end get_input_fd() */


int	get_reply_message(mtext, server, key)
char		*mtext,
		*key;
FUNC_FILE	server;
/*
	IN:   mtext	A string to hold the message to be replied.
	      server	A function server structure for the function server that
			the driver request was made to.  This contains the input			file descriptor to get the reply message from.
	      key	The transaction key of the request to reply to.

	OUT:  Returns:	The reply to be sent to the driver in mtext.
			0, if the reply is formed correctly.
			-1, if an error occured.

	PURPOSE:  This function creates a reply for the server to send as a
		  response to the driver.  It takes the file descriptor in the
		  sever structure that it recieved and uses it to read a reply
		  line from the proper input file to use in the request.  The
		  reply line is parsed and then reassembled in mtext as is,
		  except for the transaction key field, which is given the value
		  of the current request's transaction key.
*/
{
char	str[MAXMSGLEN], temp[MAXMSGLEN];
int	a;

	memset(mtext, 0, MAXMSGLEN);
	memset(str, 0, MAXMSGLEN);
	memset(temp, 0, MAXMSGLEN);

	if (!server.fd)
	{
		fprintf(stderr, "Error:  Input file fd not legal.\n\n");
		return -1;
	}

	if (fgets(str, MAXMSGLEN, server.fd) == NULL)
	{
		rewind(server.fd);
		fgets(str, MAXMSGLEN, server.fd);
	}

	if (!strlen(str))
		return -1;

	for(a = 1; a <= REQU_NUM_PARMS; a++)
	{
		svr_get_field(a, str, temp);

		if (a != REQU_TK)
			strgcat(mtext, temp);
		else
			strgcat(mtext, key);

		strgcat(mtext, sep);
	}

	return 0;

} /* end get_reply_message() */


int	get_bad_reply(mtype, mtext)
long	*mtype;
char	*mtext;
/*
	IN:   mtype	The message type of the reply message.
	      mtext	A string to hold the reply message.

	OUT:  Returns:	The reply message in mtext and the message type
			in mtype.
			0, if the reply was successfully created.
			-1, if an error occured.

	PURPOSE:  This function generates a bad reply message and returns
		  it to the caller.  It randomly selects one of five hard
		  coded reply messages to be used and fills in mtype and
		  mtext accordingly.
*/
{
int	a;

#ifdef IBM_AS400
#pragma convert(500)
#endif
        char    OCT12 = '\012';
#ifdef IBM_AS400
#pragma convert(819)
#endif

	a = wrap_rnd_ri(5);

	switch(a)
	{
		case 0: /* DDC_INIT */
		{
			(*mtype) = DDC_INIT;
			sprintf(mtext, "ddc_drv,DDC_INIT,0,0%c", OCT12);

			break;
		}
		case 1: /* DDC_NUMBER */
		{
			(*mtype) = DDC_NUMBER;
                        sprintf(mtext, "grp1,DDC_NUMBER,DDC_MENU,5%c", OCT12);

			break;
		}
		case 2: /* DDC_MENU */
		{
			(*mtype) = DDC_MENU;
			sprintf(mtext, "grp1,DDC_MENU,,Enter Hours,process,\
				menu000,tuddc1219s000,enter_hours%c", OCT12);

			break;
		}
		case 3: /* DDC_FUNCTION */
		{
			(*mtype) = DDC_FUNCTION;
			sprintf(mtext, "grp1,DDC_FUNCTION,enter_hours,\
				tuddc1219s000,FS3,1,Enter Unit Code,DX11,2,\
				1000000000,3,,S,3,0,,,tuddc1221s000,check_units,\
				,,0,300,5%c", OCT12);

			break;
		}
		case 4: /* DDC_COMBINED_FIELD */
		{
			(*mtype) = DDC_COMBINED_FIELD;
			sprintf(mtext, "ddc_drv,DDC_COMBINED_FIELD,bargrp1,\
				add_units,tuddc1210s000,FS01,1%c", OCT12);

			break;
		}
	}

	return 0;

} /* end get_bad_reply() */

int	main( argc, argv )
int	argc;
char	*argv[];
{
	int i, num;
	char	*pars[4], *ddc_id, id[20], *debug_file, *envstr;
	char	buf[MAXMSGLEN+128], 
			helpvar1[256], helpvar2[256], helpvar3[256];
	struct  tm *now;
	FILE	*fd = NULL;
	QUE_INFO	*que;

#ifdef EBCDIC
    __argvtoascii_a(argc, argv);
#endif

	ddc_id = (&(id[0]));
	/* Catch signal 15 - clean up */
        signal(SIGTERM, clean_kill);

	if (argc >=6 )
	{
		ddc_initiate = 1;
	}

	/* Get separation character to be used. */
	envstr = (char *) calloc(1, sizeof(char) * 1);
	envstr = getvar("SEPARATION_CHARACTER");
	if (!envstr)
	{
		if (argc >= 5)
		{
			strcopy(sep, argv[4]);
		}
		else
		{

			strcopy(sep, ",");
		}

		fprintf(stderr, "No separation character given, ");
		fprintf(stderr, "defaulting to %s.\n\n", sep);

	} else {
		strcopy(sep, envstr);

		fprintf(stderr, "Separation character is %s.\n\n", sep);
	}

	/* Open up message logging file - if DDC_TEST is defined */
	debug_file = getvar("DDC_TEST_SERVER");

	if (debug_file)
	{
		ddc_debug = LOGGING_ON; 
		debug_fd = fopen(debug_file, "a+");

		/* Set time of test run - write to file		*/ 
		get_time(&now);

        	sprintf(buf, "%02d-%02d-%02d[%02d:%02d:%02d]: %s",
			now->tm_year % 100, now->tm_mon, now->tm_mday,
			now->tm_hour, now->tm_min, now->tm_sec, "Start test\n");
		fwrite(buf, strlen(buf)+1, 1, debug_fd);
		fflush(debug_fd);
	}

	/* Transform 'message_queue_in' and 'message_queue_out' from
	 * the perspective of the ddc_server				*/

	if (argc >= 2)
	{
		strcopy( helpvar1, argv[1] );
	}
	else
	{		
		strcopy( helpvar1, "ddc.in" );
	}

	if (argc >= 3)
	{
		strcopy( helpvar2, argv[2] );
	}
	else
	{
		strcopy( helpvar2, "ddc.out" );
	}

	if (argc >= 4)
	{
		strcopy( helpvar3, argv[3] );
	}
	else
	{
		strcopy( helpvar3, "ddc.lvf");
	}

	pars[1] = helpvar1;
	pars[2] = helpvar2;
	pars[3] = helpvar3;

	msgid_in = ddc_create_server_queue( pars[1] );

	fprintf( stderr, "Server: Idin    : %d\n", msgid_in);

	que = ddc_open_driver_queues ( pars[2], pars[3], ddc_initiate );

	msgid_out = que->id_out;
	msgid_LVF = que->id_LVF;

	fprintf( stderr, "Server: Idout   : %d\n", msgid_out);
	fprintf( stderr, "Server: LVFIdout: %d\n", msgid_LVF);

	if(TESTLVFS)
	{
		generate_LVF_file(fd);
		fd = fopen("LVFfile", "r");
	}

	memset(msgout.mtext, 0, sizeof(msgout.mtext));
	msgout.mtype = DDC_INIT;
	/* Send menus and functions */
	strcopy( msgout.mtext, "ddc_driver");
	strgcat( msgout.mtext, sep);
	strgcat( msgout.mtext, "DDC_INIT" );
	strgcat( msgout.mtext, sep);
	strgcat( msgout.mtext, "1");
	strgcat( msgout.mtext, sep);
	strgcat( msgout.mtext, "1");
	strgcat( msgout.mtext, sep);

	if ( srv_msgsnd( msgid_out, &msgout, strlen(msgout.mtext)+1, 0 ) != 0) {
		fprintf( stderr, "Server: Message not sent.\n" );
		exit( -1 );
	}

	srv_write_mesg(SERVER_SENT, msgout.mtext);

	srv_msgrcv( msgid_in, &msgin, MAXMSGLEN, DDC_LIST_START, DDC_WAIT ); 
	srv_write_mesg(SERVER_REC, msgin.mtext);

	msgout.mtype = DDC_ACK;
	strcopy( msgout.mtext, "ddc_server" );
	strgcat( msgout.mtext, (char *) sep);
	strgcat( msgout.mtext, "DDC_ACK");
	strgcat( msgout.mtext, (char *) sep);
	srv_msgsnd( msgid_out, &msgout, strlen(msgout.mtext)+1, 0);
	fprintf( stderr, "Server: Sent: msgout.mtext = '%s'\n", msgout.mtext);
	srv_write_mesg(SERVER_SENT, msgout.mtext);
	
	/* Nu mogen listelementen ontvangen worden. */

	fprintf( stderr, "Server: Ready to receive list_elem messages.\n" );

	get_group_list();

	msgout.mtype = DDC_ACK;
	strcopy( msgout.mtext, "ddc_server" );
	strgcat( msgout.mtext, (char *) sep);
	strgcat( msgout.mtext, "DDC_ACK");
	strgcat( msgout.mtext, (char *) sep);
	srv_msgsnd( msgid_out, &msgout, strlen(msgout.mtext)+1, 0);
	fprintf( stderr, "Server: Sent: msgout.mtext = '%s'\n", msgout.mtext);
	srv_write_mesg(SERVER_SENT, msgout.mtext);

	/* Select groups, the 'grp_sel'-list will be sent back now */
	msgout.mtype = DDC_LIST_START;
	strcopy( msgout.mtext, "ddc_server" );
	strgcat( msgout.mtext, (char *) sep);
	strgcat( msgout.mtext, "DDC_LIST_START");
	strgcat( msgout.mtext, (char *) sep);
	strgcat( msgout.mtext, "grp_sel");
	strgcat( msgout.mtext, (char *) sep);
	strgcat( msgout.mtext, (char *) sep);
	srv_msgsnd( msgid_out, &msgout, strlen(msgout.mtext)+1, 0 );
	srv_write_mesg(SERVER_SENT, msgout.mtext);

	fprintf( stderr, "Server: WAITING for a MESSAGE\n" );
	srv_msgrcv( msgid_in, &msgin, MAXMSGLEN, DDC_ACK, DDC_WAIT );
	fprintf(stderr, "Server: ACKnowledgement received. ");
	fprintf(stderr, "Sending grp_sel list now.\n");
	srv_write_mesg(SERVER_REC, msgin.mtext);

	if (!send_selected_groups())
	{
		fprintf(stderr, "Server:  Groups selected and list sent to ");
		fprintf(stderr, "driver.\n\n");
	}
	else
		fprintf(stderr, "Server:  Error:  No groups selected.\n\n");

	/* Send DDC_LIST_END message to terminate group selection. */
	msgout.mtype = DDC_LIST_END;
	strcopy( msgout.mtext, "ddc_server" );
	strgcat( msgout.mtext, (char *) sep);
	strcopy( msgout.mtext, "DDC_LIST_END" );
	strgcat( msgout.mtext, (char *) sep);
	strgcat( msgout.mtext, "grp_sel" );
	strgcat( msgout.mtext, (char *) sep);
	strgcat( msgout.mtext, (char *) sep);
	srv_msgsnd( msgid_out, &msgout, strlen(msgout.mtext)+1, 0 );
	srv_write_mesg(SERVER_SENT, msgout.mtext);

	fprintf( stderr, "Server: Grp_sel list sent.\n" );
	fprintf( stderr, "Server: WAITING for a MESSAGE\n" );
	srv_msgrcv( msgid_in, &msgin, MAXMSGLEN, DDC_ACK, DDC_WAIT );
	fprintf(stderr, "Server: ACKnowledgement received, ");
	fprintf(stderr, "sending MENUs now.\n");
	srv_write_mesg(SERVER_REC, msgin.mtext);

	/* Send menu structures now. */
	send_menus();

	/* Menu structures sent.  Begin execution. */
	msgout.mtype = DDC_EXECUTE;
	strcopy( msgout.mtext, "ddc_server,DDC_EXECUTE" );
	srv_msgsnd( msgid_out, &msgout, strlen(msgout.mtext)+1, 0 );
	fprintf( stderr, "Server: I have send an EXEC message.\n" );
	srv_write_mesg(SERVER_SENT, msgout.mtext);

	fprintf(stderr, "Server: WAITING for a MESSAGE\n" );
	srv_msgrcv(msgid_in, &msgin, MAXMSGLEN, DDC_ACK, DDC_WAIT ); 
	fprintf(stderr, "Server: Text of msgin: %s\n", msgin.mtext );
	fprintf(stderr, "Server: Waiting for requests, sending replies and");
	fprintf(stderr, "lvfs now.\n");
	srv_write_mesg(SERVER_REC, msgin.mtext);

	sleep(1);
/* Ready for communication phase */
/* Loop for infinit messages waiting for requests and sending replies & LVFs */ 

	num = 100;
	fprintf(stderr, "Server: Number of messages to receive: %d\n", num );
	fprintf(stderr, "        TEST_UNEXPECTED_EXIT = %d\n",
		TEST_UNEXPECTED_EXIT);
	fprintf(stderr, "            TEST_BAD_REPLIES = %d\n\n",
		TEST_BAD_REPLIES);

	for( i=0; i<num;i++) {
		int 	bytes;
		char 	func_serv[20], func_name[14], trans_pre[5],
			trans_key[14];
		FUNC_FILE	server;

		if (i == 99)
		{
			switch(TEST_UNEXPECTED_EXIT)
			{
				case 0: /* Continue running. */
				{
					i = 0;
					break;
				}
				case 1: /* Send DDC_END_REQ message and
					   continue.                    */
				{
					msgout.mtype = DDC_END_REQ;
					strcopy(msgout.mtext, "ddc_server");
					strgcat(msgout.mtext, sep);
					strgcat(msgout.mtext, "DDC_END_REQ" );
					strgcat(msgout.mtext, sep);
					srv_msgsnd(msgid_out, &msgout,
						strlen(msgout.mtext)+1, 0 );
					fprintf(stderr, "Server: I have sent");
					fprintf(stderr, "an END message.\n");
					srv_write_mesg(SERVER_SENT, msgout.mtext);

					i = 0;
					break;
				}
				case 2: /* Exit loop. */
				{
					break;
				}
				default: /* Continue running. */
				{
					i = 0;
					break;
				}
			}
		}
					
		msgin.mtype = DDC_REPLY;

		bytes = srv_msgrcv( msgid_in, &msgin, MAXMSGLEN, 0, DDC_WAIT );

		/* Exit if the in message was not a DDC_REQUEST */
		if ( ! (msgin.mtype == DDC_REQUEST ||
			msgin.mtype == DDC_LIST_REPEAT)) 
		{
			break;
		}

		if (bytes > 0)
		{
			/* Count iteration if request is received */
			strcopy(ddc_id, "");
			strcopy(trans_key, "");
			strcopy(func_serv, "");
			strcopy(func_name, "");
			strcopy(trans_pre, "");

			svr_get_field(1, msgin.mtext, ddc_id);
			svr_get_field(6, msgin.mtext, trans_key);
			svr_get_field(11, msgin.mtext, func_serv);
			svr_get_field(12, msgin.mtext, func_name);
			svr_get_field(13, msgin.mtext, trans_pre);
			srv_write_mesg(SERVER_REC, msgin.mtext);
			fprintf( stderr, "Server: Request '%s' received\n",
				msgin.mtext );
	
			server.sname = func_serv;
			server.fname = func_name;
			server.trans_pre = trans_pre;

		}
		else
		{
			sleep (1);
		}

		if (bytes > 0)
		{
			if (!get_input_fd(&server))
			{
				if ((TEST_BAD_REPLIES) &&
				    (wrap_rnd_ri(100) >= 90))
				{
					get_bad_reply(&msgout.mtype,
						      msgout.mtext);

					fprintf(stderr, "Server: Text of reply");
					fprintf(stderr, ": '%s'\n",
						msgout.mtext);
				
					srv_msgsnd(msgid_out, &msgout,
					       strlen(msgout.mtext) + 1, 0);
					srv_write_mesg(SERVER_SENT, msgout.mtext);
				}

				msgout.mtype = msgin.mtype;
				get_reply_message(msgout.mtext, server,
						  trans_key);

			} else {
				msgout.mtype = msgin.mtype;

				sprintf(msgout.mtext,
				        "%s%sDDC_REPLY%s1%s1%s9%s%s%s1%s%s%s%s%s\
					%s%s%s%s%sError:  Bad server name%s",
					ddc_id, sep, sep, sep, sep, sep,
					trans_key, sep, sep, sep, sep, sep,
					func_serv, sep, func_name, sep,
					trans_pre, sep, sep);
			}
			
			fprintf(stderr, "Server: Text of reply: '%s'\n",
				msgout.mtext);
			srv_msgsnd(msgid_out, &msgout, strlen(msgout.mtext) + 1,
			       0);
			srv_write_mesg(SERVER_SENT, msgout.mtext);
		}
		
		if ((bytes > 0) && (!LVF_file_empty))
		{
			/* Send an LVF 10 percent of the time. */
			if (TESTLVFS && wrap_rnd_ri(100) > 90)
			{
 				send_LVF(fd);
			}	
		}
	}
	
	sleep (1);

	/* If we are exiting normally or because we are testing a forced
	   shutdown of the server, exchange DDC_END_REQ and DDC_END messages
	   with the driver.  Otherwise, skip these messages and simply
	   shut down the message queues.                                     */
	
	if (TEST_UNEXPECTED_EXIT != 2)
	{
		msgout.mtype = DDC_END_REQ;
		strcopy( msgout.mtext, "ddc_server");
		strgcat( msgout.mtext, sep);
		strgcat( msgout.mtext, "DDC_END_REQ" );
		strgcat( msgout.mtext, sep);
		srv_msgsnd( msgid_out, &msgout, strlen(msgout.mtext)+1, 0 );
		fprintf( stderr, "Server: I have sent an END message.\n" );
		srv_write_mesg(SERVER_SENT, msgout.mtext);

		sleep (1);

		srv_msgrcv( msgid_in, &msgin, MAXMSGLEN, DDC_END, DDC_WAIT ); 
		srv_write_mesg(SERVER_REC, msgin.mtext);
	}

	/* Shut down the message queues now. */

#ifndef __WIN32__
	msgctl( msgid_in, IPC_RMID, (struct msqid_ds *) 0 );
	msgctl( msgid_out, IPC_RMID, (struct msqid_ds *) 0 );
	msgctl( msgid_LVF, IPC_RMID, (struct msqid_ds *) 0 );
#endif

	fprintf( stderr, "Server: Removed the queues now.\n" );

	free_group_list();
	remove_function_server_list();
/*	fclose(fd); */

	sleep (1);

	unlink(pars[1]);
	unlink(pars[2]);
	unlink(pars[3]);

	return (0);
}
