
/*****************************************************************************
* Project	: Distributed Data Collection System ddc.driver
* File		: ddc_tools.c
* Description	: Library of special driver tools.
*
* This file contains the functions:
*  int		get_time();
*  int		get_field();
*  int		initialize_global_variables();
*  int		initialize_lvf();
*  int		initialize_LVF_element();
*  int		initialize_lvf_repeat();
*  int		initialize_request();
*  int		initialize_reply();
*  int		initialize_parameter_structure();
*  int		clear_LVF_header();
*  int		clear_LVF_element();
*  int		clear_request();
*  int		how_lvf_data();
*  int		show_lvf_data_element();
*  int		show_lvf_repeat();
*  int		show_request_data();
*  int		write_mesg();
*  int		mesg_print();
*  int		open_input_files();
*  int		close_input_files();
*  int		which_input_file();
*  int		get_hash_key();
*  int		hash_action();
*  char		get_letter();
*  int		generate_date_string();
*  int		generate_parameter();
*  int		generate_request();
*  int          strcopy();
*  int          strgcat();
*
*****************************************************************************/

/*****************************************************************************
*	System include files
*****************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#if _WIN32
# include <windows.h>
#include <sys/timeb.h>
#else
#include <sys/ipc.h>
#include <sys/timeb.h>
#include <signal.h>
#endif

/*****************************************************************************
*	Project include files
*****************************************************************************/
#include "system.h"
#include "al_proto.h"
#include "stream.h"
#include "ddc_baan.h"
#include "ddc.h"
#include "ddc_supplier.h"
#include "ddc_tools.h"
#include "al_mslot.h"
#include "al_proto.h"


int             inquiry_trans_key;
int             rndm_factor;
int             send_bad_requests;
int             bad_request;
int             trans_key;
VALID_FSERVER   *vfstable[NUM_HASH_BUCKETS];
PARM_LIST               *holding_parm;
char    holding_str[MAXMSGLEN];

#ifdef _WIN32
__declspec (dllexport) int              readreq;
__declspec (dllexport) int              open_trans;
__declspec (dllexport) int              open_lvf;
__declspec (dllexport) int              cur_requests;
__declspec (dllexport) int              logvar;
__declspec (dllexport) FILE             *debug_fd;
__declspec (dllexport) int              mode;
__declspec (dllexport) int              transaction_mode;
__declspec (dllexport) int              wait_for_inquiry;
__declspec (dllexport) int              inquiry_reply_received;
__declspec (dllexport) char             sepchar[2];
__declspec (dllexport) int              lastread;
__declspec (dllexport) int              num_infiles;
__declspec (dllexport) FILE             *infile[MAX_INPUT_FILES];
#else
int     readreq;
int     open_trans;
int     open_lvf;
int     cur_requests;
int     logvar;
FILE    *debug_fd;
int     mode;
int     transaction_mode;
int     wait_for_inquiry;
int     inquiry_reply_received;
char    sepchar[2];
int     lastread;
int     num_infiles;
FILE    *infile[MAX_INPUT_FILES];
#endif





/*****************************************************************************
*	DDC UNIVERSAL DRIVER library functions.
*****************************************************************************/

int	get_time(now)
struct tm	**now;
/*
	OUT:	A pointer to struct tm.

	PURPOSE:  This function gets the localtime.  It makes calls to time()
		  and localtime() and returns the address of a time structure
		  containing the current time.
*/
{
time_t	clocktime, *clockptr;
struct tm	*curtime;

	time(&clocktime);
	clockptr = &clocktime;
	curtime = localtime(clockptr);
	*now = curtime;

	return 0;

} /* end get_time() */


int	get_field(str,field)
char	**str;
char	*field;
/*
	IN:   str	The address of a pointer to a message string.
	      field	The address of a character string.

	OUT:  str	If there are more fields in the message, this is set
			to point to the first byte of the next message field.
	      field	The current field is copied into this string.
	      returns:  the length of field, if the field was copied correctly.
			-1, if an error occured.

	PURPOSE:  This function gets the next field of a message and returns
		  it to the caller.  If an error occurs, this function returns
		  -1.  If there are no errors, this function returns the length
		  of the field string.  Also, the str pointer is set to point
		  to the first byte of the next field, and the field pointer
		  points to the current field.
*/
{
	register	int	len = 0;
	char		*p1, *pf, sepval;

	sepval = ',';

	/* Returns: -1 or length of field	*/

	if ( str && *str && field ) {
		p1 = *str;
		pf = field;
        } else
                return( -1 );

	/* p1 points to begin of string fnum,		*/ 
	/* p2 points to first comma in that string	*/

	while ( *p1 != '\0' ) {    /* This while loop copies the contents of */
		if (*p1 == sepval) /* str into field until either the end of */
			break;     /* str or a separation character is read. */

		*pf++ = *p1++;
		len++;
	}
	*pf = '\0';
	if ( *p1 != '\0' ) p1++;  /* If not at the end of str, go to next byte. */
	*str = p1;

	return( len );

} /* end get_field() */


void	initialize_global_variables()
/*
	PURPOSE:  This function initializes the global variables that
		  will be used by the universal driver and its tools
		  library functions.
*/
{
	logvar = LOGGING_OFF;
	readreq = 0;
	open_trans = 0;
	open_lvf = 0;
	mode = 1;
	trans_key = 10000;
	cur_requests = 0;
	send_bad_requests = 1;
	bad_request = 0;
	wait_for_inquiry = 0;
	inquiry_reply_received = 0;

} /* end initialize_global_variables() */


int	initialize_lvf( lvf )
LVF_HEADER	**lvf;
/*
	IN:   lvf	A pointer to a pointer to an lvf header.

	OUT:  returns:	0, if the new lvf is initialized properly.
			-1, if an error occured.

	PURPOSE:  This function allocates space for the fields of a lvf header.
*/
{
LVF_HEADER 	*tmp;

	tmp = (LVF_HEADER *) calloc(1, sizeof(LVF_HEADER));
	tmp->LVFname = (char *) calloc(1, sizeof(char) *_LVFNAME);
	tmp->stime = (char *) calloc(1, sizeof(char) *_DDCTIME);
	*lvf = tmp;
	return 0;

} /* end initialize_lvf() */


int initialize_LVF_element( elem, esize )
LVF_FORM	**elem;
int		esize;
/*
	IN:   elem	A pointer to a pointer to a LVF_FORM.
	      esize	The size to be used for the element's record string.

	OUT:  returns	0, if the element was initialized properly.
			-1, if an error occured.

	PURPOSE:  This function allocates space for each field of an element.
*/
{
LVF_FORM	*rptr;

	rptr = (LVF_FORM *) calloc (1, sizeof(LVF_FORM));
	rptr->listid = (char *) calloc (1, sizeof(char)*_LISTID);
	rptr->record = (char *) calloc (1, sizeof(char)*esize);
	rptr->action = (char *) calloc (1, sizeof(char)*2);

	*elem = rptr;
	return 0;

} /* end initialize_LVF_element() */


int	initialize_lvf_repeat( lvf )
LVF_REPEAT_FORM	**lvf;
/*
	IN:   lvf	A pointer to a pointer to a LVF_REPEAT_FORM.

	OUT:  Returns:	0, if the repeat form was initialized properly.
			-1, if an error occured.

	PURPOSE:  This function allocates space for each field of a LVF
		  repeat form.
*/
{
LVF_REPEAT_FORM *tmp;

	tmp = (LVF_REPEAT_FORM *) calloc(1, sizeof(LVF_REPEAT_FORM));
	tmp->ddc_id = (char *) calloc(1, sizeof(ID_LEN));
	tmp->LVFname = (char *) calloc(1, sizeof(_LVFNAME));
	tmp->dtime = (char *) calloc(1, sizeof(_DDCTIME));
	*lvf = tmp;
	return 0;

} /* end initialize_lvf_repeat() */


int	initialize_request(generic_request)
REQUEST_FORM	**generic_request;
/*
	IN:   generic_request	A pointer to a pointer to a REQUEST_FORM.

	OUT:  returns:	0, if the request is initialized correctly.
			-1, if an error occured.

	PURPOSE:  This function allocates space for the fields of a REQUEST_FORM.
*/
{
REQUEST_FORM	*request;
int	i;

	request = ((REQUEST_FORM *) 0 );
	/* Allocate request form */
	request = (REQUEST_FORM *) calloc(1, sizeof(REQUEST_FORM));
	request->ddc_id = (char *) calloc(1, sizeof(ID_LEN));
	request->dname = (char *) calloc(1, sizeof(char) * _FNAME);
	request->fname = (char *) calloc(1, sizeof(char) * _FNAME);
	request->device_date = (char *) calloc(1, sizeof(char) * _DDCTIME);
	request->contr_date = (char *) calloc(1, sizeof(char) * _DDCTIME);
	request->trans_key = (char *) calloc(1, sizeof(char) * _TRANSKEY);
	request->trans_prefix = (char *) calloc(1,sizeof(char)* _TRANPREFIX);
	request->trans_value = (char *) calloc(1, sizeof(char) * _TRANSKEY);
	request->reserved1 = (char *) calloc(1, sizeof(char) * 25);
	request->reserved2 = (char *) calloc(1, sizeof(char) * 25);
	request->reserved3 = (char *) calloc(1, sizeof(char) * 25);
	request->fpars = (char **) calloc (MAX_FUN_PARMS, sizeof(char *));
	
	for (i=0; i < MAX_FUN_PARMS; i++)
	{
		request->fpars[i] = (char *) calloc (1, MAX_PARM_LEN);
	}

	if (request)
	{
		*generic_request = request;
		return 0;
	}
	else
		return -1;

} /* end initialize_request() */


int	initialize_reply(generic_reply)
REPLY_FORM	**generic_reply;
/*
	IN:   generic_reply  A pointer to a pointer to a struct REPLY_FORM.

	OUT:  returns:	0, if the reply is initialized correctly.
			-1, if an error occured.

	PURPOSE:  This function allocates space for the fields of a REPLY_FORM.
*/
{
REPLY_FORM	*reply;

	if (!generic_reply)
		return -1;

	reply = ((REPLY_FORM *) 0 );
	reply = (REPLY_FORM *) calloc(1, sizeof(REPLY_FORM));
	reply->ddc_id = (char *) calloc(1, sizeof(char)* ID_LEN);
	reply->fserv = (char *) calloc(1, sizeof(char)* _FSERV);
	reply->fname = (char *) calloc(1, sizeof(char)* _FNAME);
	reply->trans_key = (char *) calloc(1, sizeof(char)* _TRANSKEY);
	reply->trans_prefix = (char *) calloc(1, sizeof(char)* _TRANPREFIX);
	reply->trans_value = (char *) calloc(1, sizeof(char)* _TRANSKEY);
	reply->err_mess = (char *) calloc(1, 30);

	if (reply)
	{
		*generic_reply = reply;
		return 0;
	}
	else
		return -1;

} /* end initialize_reply() */


void	initialize_parameter_structure(parm)
PARM_LIST	*parm;
/*
	IN:   parm	A pointer to a PARM_LIST structure.

	OUT:  Returns:	0, if parm was initialized properly.
			-1, if an error occured.

	PURPOSE:  This function initializes the fields of a PARM_LIST
		  structure.
*/
{
	parm->ptype = ' ';
	memset(parm->pvalue, 0, MAX_PARM_LEN);
	parm->pseqnr = 0;
	parm->parmlen = 0;
	parm->dec = 0;
	parm->just = 0;
	parm->zfill = 0;
	parm->parmtype = 0;
	parm->onetime = 0;
	memset(parm->listid, 0, MAX_PARM_LEN);
	memset(parm->bitmask, 0, MAX_PARM_LEN);
	memset(parm->template, 0, MAX_PARM_LEN);
	memset(parm->iserver, 0, MAXNAMELEN);
	memset(parm->iname, 0, MAXNAMELEN);
}


int	clear_LVF_header( lvf )
LVF_HEADER *lvf;
/*
	IN:   lvf	A pointer to a lvf header structure.

	OUT:  returns:	0, if the fields of the specified header were cleared.
			-1, if an error occured.

	PURPOSE:  This function simply clears the values of each field of a
		  specified lvf header.
*/
{
	if (!lvf)
		return -1;

	strcopy(lvf->LVFname, "");
	strcopy(lvf->stime, "");
	lvf->listnum = 0;
	lvf->storage_loc = 0;

	return 0;

} /* end clear_LVF_header() */


int	clear_LVF_element( element )
LVF_FORM *element;
/*
	IN:   element	A pointer to an object of type LVF_FORM.

	OUT:  returns:	0, if the fields of the element were cleared.
			-1, if an error occured.

	PURPOSE:  This function simply clears the values of each field of a
		  specified element.
*/
{
	if (!element)
		return -1;

	strcopy(element->listid, "");
	strcopy(element->record, "");
	strcopy(element->action, "");

	return 0;

} /* end clear_LVF_element() */


int	clear_request(generic_request)
REQUEST_FORM	**generic_request;
/*
	IN:   generic_request	A pointer to a pointer to a REQUEST_FORM.

	OUT:  returns:	0, if specified request is cleared.
			-1, if an error occured.

	PURPOSE:  This function clears the data out of the fields of a request.
*/
{
REQUEST_FORM	*request;
int	i;

	request = *generic_request;

	if (!(request))
		return -1;

	/* Allocate request form */
	memset(request->ddc_id, 0, sizeof(ID_LEN));
	memset(request->dname, 0, sizeof(char)* _FNAME);
	memset(request->fname, 0, sizeof(char)* _FNAME);
	memset(request->device_date, 0, sizeof(char)* _DDCTIME);
	memset(request->contr_date, 0, sizeof(char)* _DDCTIME);
	memset(request->trans_key, 0, sizeof(char)* _TRANSKEY);
	memset(request->trans_prefix, 0, sizeof(char)* _TRANPREFIX);
	memset(request->trans_value, 0, sizeof(char)* _TRANSKEY);
	memset(request->reserved1, 0, sizeof(char)* 25);
	memset(request->reserved2, 0, sizeof(char)* 25);
	memset(request->reserved3, 0, sizeof(char)* 25);
	
	for (i=0; i < MAX_FUN_PARMS; i++)
	{
		memset(request->fpars[i], 0, sizeof(char) * MAX_PARM_LEN);
	}

	*generic_request = request;
	return 0;

} /* end clear_request() */


int	show_lvf_data( lh )
LVF_HEADER	*lh;
/*
	IN:   lh	A pointer to a LVF_HEADER.

	OUT:  Returns:	0, if the LVF data is displayed without problems.
			-1, if an error occures.

	PURPOSE:  This function displays the data for an LVF message.  First,
		  it checks to make sure that lh is not NULL.  If it is, this
		  function returns -1.  If lh is not NULL, the data is displayed
		  using a call to mesg_print() and 0 is returned.
*/
{
	if ( ! lh ) {
		fprintf( stderr, "A LVFname does not exist!\n" );
		return( -1 );
	}

	fprintf( stderr, "Child: ShowLVFdata.LVFname: '%s'\n\n", lh->LVFname );
	mesg_print(DBG_MSG_30, lh->LVFname, "", "", 0, 0, MSG_FORMAT_5,
		   DRIVER_REC);

	return 0;

} /* end show_lvf_data() */


int	show_lvf_data_element( elem )
LVF_FORM	*elem;
/*
	IN:   elem	A pointer to a LVF_FORM.

	OUT:  Returns:	0, if the lvf element data was displayed properly.
			-1, if an error occured.

	PURPOSE:  This function displays the data of an LVF element.  First, it
		  checks to see if elem is NULL.  If it is, an error message is
		  printed and -1 is returned.  If elem is not NULL, the LVF
		  element's data is displayed using calls to mesg_print() and
		  0 is returned.
*/
{
	if ( ! elem ) {
		fprintf( stderr, "A LVFname does not exist!\n\n" );
		return( -1 );
	}

	fprintf( stderr, "Child: ShowLVFdata.LVFname: '%s'\n\n", elem->listid );
	fprintf( stderr, "Child: ShowLVFdata.element: %s\n\n", elem->record );
	fprintf( stderr, "Child: ShowLVFdata.action: %s\n\n", elem->action );

	mesg_print(DBG_MSG_31, elem->listid, DBG_MSG_32, elem->record, 0, 0,
		   MSG_FORMAT_6, DRIVER_REC);
	mesg_print(DBG_MSG_33, elem->action, "", "", 0, 0, MSG_FORMAT_5,
		   DRIVER_CNT);

	return 0;

} /* end show_lvf_data_element() */


int	show_lvf_repeat( lr )
LVF_REPEAT_FORM	*lr;
/*
	IN:   lr	A pointer to a LVF_REPEAT_FORM.

	OUT:  Returns:	0, if the data for the repeat form was displayed properly.
			-1, if an error occured.

	PURPOSE:  This function displays the data for an LVF repeat message.
		  First, it checks to see if lr is NULL.  If it is, an error
		  message is printed and -1 is returned.  If lr is not NULL, the
		  repeat forms data is displayed using calls to mesg_print() and
		  0 is returned.
*/
{
	if ( ! lr ) {
		fprintf( stderr, "A LVFname does not exist!\n" );
		return( -1 );
	}

	fprintf( stderr, "ShowLVF repeat data.LVFname: '%s'\n", lr->LVFname );

	mesg_print(DBG_MSG_34, lr->LVFname, "", "", 0, 0, MSG_FORMAT_7,
		   DRIVER_SENT);
	mesg_print(DBG_MSG_35, lr->dtime, "", "", 0, 0, MSG_FORMAT_7,
		   DRIVER_CNT);
	mesg_print(DBG_MSG_36, "", "", "", lr->dtype, 0, MSG_FORMAT_8,
		   DRIVER_CNT);

	return 0;

} /* end show_lvf_repeat() */


void	show_reply_data( rep )
REPLY_FORM	*rep;
/*
	IN:  rep	A pointer to a reply form.

	OUT: returns	0, if the reply is displayed correctly.
			-1, if an error occurs.

	PURPOSE:  This function displays the data that makes up the given
		  reply.  This function sends the reply data to stderr
		  and also calls write_mesg() so that the message can be
		  used for debugging.
*/
{
	char *x;

	x = (char *) calloc(1, sizeof(char) * MAXMSGLEN);

	fprintf( stderr, "Reply data was: %s = DDC_ID, %d = mode, %d = errno, \
			%d = err_parm, %s = fun. server, %s = fun. name, %s \
			= err mesg. \n", rep->ddc_id, rep->state, rep->err_no,
			rep->err_par, rep->fserv, rep->fname, rep->err_mess);

	sprintf( x, "%s%sDDC_REPLY%s%d%s%d%s%s%s%d%s%s%s%s%s%s%s%s\n", 
			rep->ddc_id,
			sepchar, sepchar, rep->state, sepchar, rep->err_no,
			sepchar, rep->trans_key, sepchar, rep->err_par, sepchar,
			rep->fserv, sepchar, rep->fname, sepchar,
			rep->trans_prefix, sepchar, rep->err_mess);
	write_mesg(DRIVER_REC, x);

	if (atoi(rep->trans_key) == inquiry_trans_key)
	{
		wait_for_inquiry = 0;
		inquiry_reply_received = 1;
	}

	free(x);

} /* end show_reply_data() */


int	show_request_data( req )
REQUEST_FORM	*req;
/*
	IN:  req	A pointer to a request form.

	OUT: returns	0, if the request is displayed correctly.
			-1, if an error occurs.

	PURPOSE:  This function displays the data that makes up the given
		  request.  This function sends the request data to stderr
		  and also calls write_mesg() so that the message can be
		  used for debugging.
*/
{
	int i;
	char *x;

	x = (char *) calloc(1, sizeof(char) * MAXMSGLEN);

	fprintf( stderr, "Request data was: %s = DDC_ID, %d = mode, \
			%s = dev. date, %s = contr. date, %s = tran. key,\
			%s = fun. name, %s = func. server \
			%s = trans. prefix. \n", req->ddc_id, req->state,
			req->device_date, req->contr_date,
			req->trans_key, req->dname, req->fname, 
			req->trans_prefix);
	sprintf( x, "%s%sDDC_REQUEST%s%d%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
		   req->ddc_id, sepchar, sepchar, req->state, sepchar,
		   req->contr_date, sepchar, req->device_date, sepchar,
		   req->trans_key, sepchar, sepchar, sepchar, sepchar, sepchar,
		   req->dname, sepchar, req->fname, sepchar, req->trans_prefix);

	for (i=0; i<req->nr_of_fpars; i++) 
	{
		fprintf( stderr, "Parameter %d - %s \n", i+1, req->fpars[i]);
		strgcat(x, sepchar);
		strgcat(x, req->fpars[i]);
	}
	strgcat(x, "\n");
	write_mesg(DRIVER_SENT, x);
	free(x);
	return 0;

} /* end show_request_data() */


int	write_mesg( who, out )
int	who; 
char	*out;
/*
	IN:   who	A code that specifies which prefix to attach to the
			message to be displayed.
	      out	The message to be displayed.

	OUT:  returns:	0, if the message was displayed correctly.
			-1, if an error occured.

	PURPOSE:  This function is used to display a message to the file used
		  to help debug the driver and server code.  First, it checks
		  whether or not the logging option is on.  If it isn't, nothing
		  is printed, and the function returns.  If the logging option
		  is on, this function sends the proper prefix, as specified by
		  who, and, then, sends the message to the debugging file.
*/
{
	if (logvar == LOGGING_ON)
	{
		if (!debug_fd)
			return -1;

		switch(who)
		{
			case DRIVER_SENT:
			{
				fprintf(debug_fd, "%s", DRIVER_SENT_TXT);
				break;
			}
			case DRIVER_REC:
			{
				fprintf(debug_fd, "%s", DRIVER_REC_TXT);
				break;
			}
			case DRIVER_CNT:
			{
				fprintf(debug_fd, "%s", DRIVER_CNT_TXT);
				break;
			}
			case DRIVER_ERR:
			{
				fprintf(debug_fd, "%s", DRIVER_ERR_TXT);
				break;
			}
		}

		fprintf(debug_fd, "%s\n", out);
		fflush(debug_fd);
	}

	return 0;

} /* end write_mesg() */


int	mesg_print(str1, str2, str3, str4, val1, val2, format, mode)
char	*str1, *str2, *str3, *str4;
int	val1, val2, format, mode;
/*
	IN:   str1 - str4   Strings to be used in forming messages.
	      val1, val2    Integer values to be used in message.
	      format	    Code used to determine message format.
	      mode	    Specifies DRIVER_SEND, DRIVER_REC, or DRIVER_ERR.

	OUT:  returns:	0, if message was printed correctly.
			-1, if an error occured.

	PURPOSE:	This function displays a message.  Several different
			functions will call this function to display information
			for the user, this function takes many arguments.  It
			looks at format to determine which arguments to place on
			the display line and in which order.  Once the message
			has been formed, it is displayed to both standard
			ouput and sent in a call to write_mesg() for use in
			debugging.
*/
{

char	*msgline;

	msgline = (char *) calloc (1, sizeof(char) * MAXMSGLEN);

	switch(format)
	{
	     case MSG_FORMAT_1:
	     {
		sprintf(msgline, "%s%s%s\n", str1, str2, str3);
		break;
	     }
	     case MSG_FORMAT_2:
	     {
		sprintf(msgline, "%s%s%d\n", str1, str2, val1);
		break;
	     }
	     case MSG_FORMAT_3:
	     {
		sprintf(msgline, "%s%s%d%s%s\n", str1, str2, val1, str3, str4);
		break;
	     }
	     case MSG_FORMAT_4:
	     {
		sprintf(msgline, "%s%s%d%s%d\n", str1, str2, val1, str3, val2);
		break;
	     }
	     case MSG_FORMAT_5:
	     {
		sprintf(msgline, "%s%s\n", str1, str2);
		break;
	     }
	     case MSG_FORMAT_6:
	     {
		sprintf(msgline, "%s%s%s%s,", str1, str2, str3, str4);
		break;
	     }
	     case MSG_FORMAT_7:
	     {
		sprintf(msgline, "%s%s", str1, str2);
		break;
	     }
	     case MSG_FORMAT_8:
	     {
		sprintf(msgline, "%s%d\n", str1, val1);
		break;
	     }
	}

	write_mesg(mode, msgline);

	free(msgline);

	return 0;

} /* end mesg_print() */


int open_input_files(num_files, nstr)
int	num_files;
char	*nstr;
/*
	IN:   num_files	The number of files to be opened.
	      nstr	The name string for the files.

	OUT:  Returns	0, if the files were opened correctly.
			-1, if an error occured.

	PURPOSE:  This function opens input files for the universal
		  driver.  It takes the number of files to be made,
		  and for each one, it uses the given file name string
		  to create a file name.  The function then opens the
		  file, using the created name, and stores the fd in
		  the infile[] array.
*/
{
char	*filename, *location, *buff = NULL;
long	a;

	filename = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
	location = (char *) calloc(1, sizeof(char) * MAXNAMELEN);

	if (num_files > MAX_INPUT_FILES)
	{
		fprintf(stderr, "Error:  Too many input files.\n\n");
		return -1;
	}

	for(a = 0; a < num_files; a++)
	{
		strcopy(filename, nstr);
	
		location = getenv("DDC_INFILE_PATH");
		fprintf(stderr, "DDC_INFILE_PATH: %s\n", location);
		fflush(stderr);
	
		buff = (char *) calloc(1, 256);
		
		sprintf(buff,"%s/%s%d",location, filename, a);

		if (!buff)
			return -1;

		strcopy(filename, buff);
	
		infile[a] = fopen(filename, "r");
		if( (infile[a] = fopen( filename, "r" )) == NULL )   {
			fprintf(stderr, "Error, Driver can't open infile\n");
			free(filename);
			return 1;
		}
	}

	free(filename);
	return 0;

} /* end open_input_files() */


int	close_input_files(num_files)
int	num_files;
/*
	IN:   num_files	The number of input files to be closed.

	OUT:  Returns	0, if the files were closed correctly.
			-1, if an error occured.

	PURPOSE:  This function closes the input files for the universal
		  driver.  It takes the number of files that were opened
		  by open_input_files() and uses that number to traverse
		  the infile[] fd array, closing the appropriate file for
		  each fd.
*/
{
int	a = 0;

	if (num_files > MAX_INPUT_FILES)
	{
		fprintf(stderr, "Error:  Too many input files.\n\n");
		return -1;
	}

	for(a = 0; a < num_files; a++)
	{
		if (fclose(infile[a]))
		{
			fprintf(stderr, "Error:  Input files did not");
			fprintf(stderr, " close correctly.");

			return -1;
		}
	}

	return 0;

} /* end close_input_files() */


int	which_input_file(infd)
FILE	**infd;
/*
	OUT:   Returns	The file descriptor of the file to read input from
			in infd;
			0, if an fd was chosen successfully.
			-1, if an error occured.

	PURPOSE:  This function is used to choose the file to be read from.
		  It allows the multiple files to be read out of order,
		  instead of reading one file until it is empty before going
		  on to the next file.
*/
{
struct tm	*now;
int		temp;

	if (!lastread)
	lastread = rndm_factor * 3;
	else
		lastread = rndm_factor / lastread;

	get_time(&now);
	rndm_factor = now->tm_sec;

	temp = (lastread * rndm_factor) + rndm_factor;
	lastread = temp % num_infiles;

	*infd = infile[lastread];

	return 0;

} /* end which_input_file() */

int	get_hash_key(item)
VALID_FSERVER	item;
/*
	IN:   item	The VALID_FSERVER item that a key is needed for.

	OUT:  Returns	A key to the valid function server hash table, or
			-1 if an error occured.

	PURPOSE:  This function takes a VALID_FSERVER structure and uses its
		  fields to generate a key to the valid function server hash
		  table.  A key is generated in one of two ways.  If the given
		  structure contains a transaction prefix, the transaction
		  prefix and the number of parameters is used to generate the
		  key.  If no transaction prefix is given, get_hash_key() uses
		  the characters at positions two through four of the server
		  name, the first three characters of the function name, and
		  the number of parameters to generate the key.  Once a key
		  has been generated, it is returned to the sender.  If an error
		  occures in generating the key, -1 is returned.
*/
{
char	buff[MAXMSGLEN], buff2[MAXMSGLEN];
int	snum = 0, fnum = 0, a, key, len;

	if (strcmp(item.trans, ""))
	{
		strcopy(buff, item.trans);
		len = strlen(buff);

		for(a = 0; a < len; a++)
		{
			key = buff[a];
			snum += key;
		}

		a = snum + item.num_parms;
		
		key = a % NUM_HASH_BUCKETS;
	} else {
		if (!strcmp(item.sname, ""))
			return -1;

		strcopy(buff, item.sname);

		for(a = 1; a < 4; a++)
			buff2[a - 1] = buff[a];

		buff2[a - 1] = '\0';

		strcopy(buff, item.fname);
		len = strlen(buff);

		if (len)
		{
			for(a = 0; a < 3; a++)
			{
				key = buff[a];
				fnum += key;

				key = buff2[a];
				snum += key;
			}
		}

		a = snum + fnum + item.num_parms;
		key = a % NUM_HASH_BUCKETS;
	}

	return key;

} /* end get_hash_key() */


int	hash_action(item, action)
VALID_FSERVER	*item;
int		action;
/*
	IN:   item	A pointer to a VALID_FSERVER structure.
	      action	An action code.

	OUT:  Returns	This varies, based on the action being performed.
		HASH_ACTION_ADD		0, if item was added correctly.
					-1, if an error occured.
		HASH_ACTION_REMOVE	0, if item was removed correctly.
					-1, if an error occured.
		HASH_ACTION_LOOKUP	0, if item was found in the table.
					-1, if item was not found.
		HASH_ACTION_RNDSEL	0, if a function server was chosen.
					-1, if an error occured.

	PURPOSE:  This function takes a VALID_FSERVER structure and uses it
		  to perform an action on the hash table based on the given
		  action code.  If the action is HASH_ACTION_ADD, this
		  function gets a hash key for item and adds it to the hash
		  table.  If the action is HASH_ACTION_REMOVE, this function
		  gets a hash key for item and looks for it on the table.  If
		  the item is in the table, it is removed, otherwise, 0 is
		  returned.  If the action is HASH_ACTION_LOOKUP, this
		  function gets a hash key for item and looks for it on the 
		  table.  If the item is in the table, 1 is returned, otherwise
		  0 is returned.  If the action is HASH_ACTION_RNDSEL, this
		  function selects one of the function servers from the hash
		  table at random and returns the chosen function server's
		  values in item.
*/
{
VALID_FSERVER	*ptr, *temp;
PARM_LIST	*tparm;
int		a, key, num;

	if ((action != HASH_ACTION_REMOVE) && (action != HASH_ACTION_RNDSEL))
	{
		key = get_hash_key(*item);

		if (key == -1)
			return -1;
	}

	switch(action)
	{
		case 1:  /* HASH_ACTION_ADD */
		{
			temp = (VALID_FSERVER *)calloc(1, sizeof(VALID_FSERVER));
			temp->group = (char *) calloc(1, sizeof(char) * _GRP_ID);
			temp->sname = (char *) calloc(1, sizeof(char) *
							   MAXNAMELEN);
			temp->fname = (char *) calloc(1, sizeof(char) *
							   MAXNAMELEN);
			temp->trans = (char *) calloc(1, sizeof(char) *
							   MAXNAMELEN);

			strcopy(temp->group, item->group);
			strcopy(temp->sname, item->sname);
			strcopy(temp->fname, item->fname);
			strcopy(temp->trans, item->trans);
			temp->num_parms = item->num_parms;
			temp->fparm = item->fparm;
			temp->lparm = item->lparm;
			temp->next = NULL;

			if (vfstable[key] == NULL)
				vfstable[key] = temp;
			else {
				ptr = vfstable[key];

				if (!strcmp(ptr->group, item->group) &&
				   !strcmp(ptr->sname, item->sname) &&
				   !strcmp(ptr->fname, item->fname) &&
				   ptr->num_parms == item->num_parms)
					break;

				while(ptr->next != NULL)
				{
					if (!strcmp(ptr->next->group,
						    item->group) &&
					   !strcmp(ptr->next->sname,
						   item->sname) &&
					   !strcmp(ptr->next->fname,
						   item->fname) &&
					   ptr->next->num_parms ==
						   item->num_parms)
						break;

					ptr = ptr->next;
				}

				ptr->next = temp;
			}

			return 0;
		}
		case 2: /* HASH_ACTION_REMOVE */
		{
			for(a = 0; a < NUM_HASH_BUCKETS; a++)
			{
				if (vfstable[a] == NULL)
					continue;

				ptr = vfstable[a];

				while(ptr != NULL)
				{
					temp = ptr;
					ptr = temp->next;

					memset(temp->group, 0, _GRP_ID);
					memset(temp->sname, 0, MAXNAMELEN);
					memset(temp->fname, 0, MAXNAMELEN);
					memset(temp->trans, 0, MAXNAMELEN);
					temp->num_parms = 0;

					while(temp->fparm != NULL)
					{
						tparm = temp->fparm;
						temp->fparm = tparm->next;
						free(tparm);
					}

					temp->lparm = NULL;
					temp->next = NULL;

					free(temp->group);
					free(temp->sname);
					free(temp->fname);
					free(temp->trans);
					free(temp);
				}
			}

			return 0;
		}
		case 3: /* HASH_ACTION_LOOKUP */
		{
			ptr = vfstable[key];

			/* If the curent request is bad intentionally for */
			/* for testing, let it pass.                      */
			if (bad_request)
			{
				bad_request = 1;
				return 0;
			}

			while(ptr != NULL)
			{
				if (strcmp(item->trans, ""))
				{
					if (!strcmp(ptr->trans, item->trans) &&
					   ptr->num_parms == item->num_parms)
						return 0;
				} else {
					if (!strcmp(ptr->sname, item->sname) &&
				   	   !strcmp(ptr->fname, item->fname) &&
				   	   ptr->num_parms == item->num_parms)
						return 0;
				}

				ptr = ptr->next;
			}

			return 0;
			
		}
		case 4: /* HASH_ACTION_RNDSEL */
		{
			a = 0;
			num = wrap_rnd_ri(NUM_HASH_BUCKETS);

			while(vfstable[num] == NULL)
			{
				if (a < 30)
				      num = wrap_rnd_ri(NUM_HASH_BUCKETS);
				else {
					a = 0;

					while(vfstable[num] == NULL)
					{
						a++;
						num++;

						if (num >= NUM_HASH_BUCKETS)
							num = 0;

						if (a >= NUM_HASH_BUCKETS)
							return -1;
					}
				}

				a++;
			}

			ptr = vfstable[num];

			if (ptr->next != NULL)
			{
				a = 0;

				while(ptr != NULL)
				{
					a++;
					ptr = ptr->next;
				}

				a = wrap_rnd_ri(a);

				ptr = vfstable[num];

				for(num = 0; num < a; num++)
					ptr = ptr->next;
			}

			strcopy(item->group, ptr->group);
			strcopy(item->sname, ptr->sname);
			strcopy(item->fname, ptr->fname);
			strcopy(item->trans, ptr->trans);
			item->num_parms = ptr->num_parms;
			item->fparm = ptr->fparm;
			item->lparm = ptr->lparm;
			item->next = NULL;

			return 0;

			break;
		}
		default:
		{
			return -1;
		}

	} /* switch() */

	return -1;

} /* end hash_action() */


char	get_letter(val)
int	val;
/*
	IN:   val	An integer value which indicates which letter to return.

	OUT:  Returns:  The letter of the alphabet indicated by val.
			NULL, if a bad value was recieved;

	PURPOSE:  This function simply takes an integer value from 0 to 25 and
		  returns a character from a to z.
*/
{
	switch(val)
	{
		case 0:
			return 'a';
		case 1:
			return 'b';
		case 2:
			return 'c';
		case 3:
			return 'd';
		case 4:
			return 'e';
		case 5:
			return 'f';
		case 6:
			return 'g';
		case 7:
			return 'h';
		case 8:
			return 'i';
		case 9:
			return 'j';
		case 10:
			return 'k';
		case 11:
			return 'l';
		case 12:
			return 'm';
		case 13:
			return 'n';
		case 14:
			return 'o';
		case 15:
			return 'p';
		case 16:
			return 'q';
		case 17:
			return 'r';
		case 18:
			return 's';
		case 19:
			return 't';
		case 20:
			return 'u';
		case 21:
			return 'v';
		case 22:
			return 'w';
		case 23:
			return 'x';
		case 24:
			return 'y';
		case 25:
			return 'z';
		default:
			return '\0';  /* NULL */
	}

} /* end get_letter() */


int	generate_date_string(len, str)
int	len;
char	*str;
/*
	IN:   len	The length of the date string to be generated.
	      str	A string to put the generated date in.

	OUT:  Returns:	The generated date string in str.
			0, if the date string was generated properly.
			-1, if an error occured.

	PURPOSE:  This function take a length and a string pointer and
		  generated a date string.  First, it checks whether or
		  not the length parameter is a valid value (6, 8, 12,
		  or 14).  If the length is not valid, a -1 is returned,
		  otherwise, the function calls get_time() to get the 
		  current date and time.  Next, the values returned by
		  get_time() are converted to string form.  Finally, the
		  values are concatonated together to form a single date
		  string of the appropriate length in str and a zero is
		  returned.
*/
{
long	year, mon, day, hour, min, sec;
char	yearstr[5], monstr[3], daystr[3], hourstr[3], minstr[3], secstr[3];
char	temp[MAX_PARM_LEN];
char	*tmpbuf;
struct tm	*now;

	if((len != 8) && (len != 12) && (len != 14))
		return -1;

	/* Clear the time and date strings. */
	memset(yearstr, 0, 5);
	memset(monstr, 0, 3);
	memset(daystr, 0, 3);
	memset(hourstr, 0, 3);
	memset(minstr, 0, 3);
	memset(secstr, 0, 3);
	memset(temp, 0, MAX_PARM_LEN);

	/* Get the current time and date. */
	get_time(&now);

	/* Get current time and date values in long form. */
	year = now->tm_year;
	mon = now->tm_mon;
	day = now->tm_mday;
	hour = now->tm_hour;
	min = now->tm_min;
	sec = now->tm_sec;

	/* Adjust year if it is needed. */
	if(len != 6)
		year = year + 1900;

	/* Convert time and date values into string form. */
	
	tmpbuf = (char *) calloc(1, 256);

	sprintf(yearstr,"%d",year);

	mon = mon + 1;

	if (mon < 10)
	{
		sprintf(monstr,"0%d",mon);
	}
	else
	{
	sprintf(monstr,"%d",mon);
	}

        if (day < 10)
	{
		sprintf(daystr,"0%d",day);
	}
	else
	{
		sprintf(daystr,"%d",day);
	}

        if (hour < 10)
	{
		sprintf(hourstr,"%d",hour);
	}
	else
	{
		sprintf(hourstr,"%d",hour);
	}

        if (min < 10)
	{
		sprintf(minstr,"0%d",min);
	}
	else
	{
		sprintf(minstr,"%d",min);
	}

        if (sec < 10)
	{
		sprintf(secstr,"0%d",sec);
	}
	else
	{
		sprintf(secstr,"%d",sec);
	}

	/* Put together the parameter string. */
	if (len == 6)
	{
		/* Add the month. */
		strncat(temp, monstr, 2);

		/* Add the day. */
		strncat(temp, daystr, 2);

		/* Add the year. */
		strncat(temp, yearstr, 4);
	}

	if (len >= 8)
	{
		/* Add the year. */
		strncat(temp, yearstr, 4);

		/* Add the month. */
		strncat(temp, monstr, 2);

		/* Add the day. */
		strncat(temp, daystr, 2);
	}

	if (len >= 12)
	{
		/* Add the hour. */
		strncat(temp, hourstr, 2);

		/* Add the minute. */
		strncat(temp, minstr, 2);
	}

	if (len == 14)
	{
		/* Add the second. */
		strncat(temp, secstr, 2);
	}

	strcopy(str, temp);

	return(0);

} /* generate_date_string() */


int	generate_parameter(parm, str)
PARM_LIST	*parm;
char		*str;
/*
	IN:   parm	A pointer to a parameter structure.
	      str	A string to store the generated parameter in.

	OUT:  Returns:	The generated parameter in str.
			0, if the parameter was generated properly.
			-1, it an error occured.

	PURPOSE:  This function takes a parameter structure and a string as input
		  and returns a generated parameter value in str.  The parameter
		  is generated according to the definition in the parameter
		  structure.
*/
{
int	a, len, val, dec;
long	bound;
char	temp[MAX_PARM_LEN], tstr[MAX_PARM_LEN];

	if ((parm->parmtype == 1) || (parm->parmtype == 2))
	{
		strcopy(str, ",");
		strgcat(str, parm->pvalue);
		return 0;

	} else if (parm->parmtype != 0)
		return -1;

	memset(str, 0, MAX_PARM_LEN);
	memset(temp, 0, MAX_PARM_LEN);
	memset(tstr, 0, MAX_PARM_LEN);

	switch(parm->ptype)
	{
		case 'S':
		case 's':                    /* The parameter is a string. */
		{
			val = (wrap_rnd_ri(parm->parmlen) + 1);

			for(a = 0; a < val; a++)
				temp[a] = get_letter(wrap_rnd_ri(26));

			temp[a] = '\0';

			strcopy(str, ",");
			strgcat(str, temp);
			return 0;
		}
		case 'N':
		case 'n':                    /* The parameter is a number. */
		{
			if (parm->dec == 0)
			{
				/* Get upper bound for the parameter value. */
				bound = 1;
				for(a = 0; a < parm->parmlen; a++)
					bound = bound * 10;

				val = wrap_rnd_ri(bound);

				sprintf(temp, "%d", val);
				strcopy(str, ",");

				if (parm->zfill == 0)
				{
					len = parm->parmlen - strlen(temp);

					for(a = 0; a < len; a++)
						strgcat(str, "0");

					strgcat(str, temp);

				} else
					strgcat(str, temp);

			} else {
				len = (parm->parmlen - (parm->dec + 1));

				if (len > 6) len = 6;

				/* Get upper bound for integer value. */
				bound = 1;
				for(a = 0; a < len; a++)
					bound = bound * 10;

				val = wrap_rnd_ri(bound);

				/* Get upper bound for decimal value. */
				bound = 1;
				for(a = 0; a < parm->dec; a++)
					bound = bound * 10;

				dec = wrap_rnd_ri(bound);

				sprintf(temp, "%d.%d", val, dec);
				strcopy(str, ",");

				if (parm->zfill == 0)
				{
					sprintf(tstr, "%d", val);
					len = len - strlen(tstr);

					for(a = 0; a < len; a++)
						strgcat(str, "0");

					strgcat(str, temp);

					memset(tstr, 0, MAX_PARM_LEN);
					sprintf(tstr, "%d", dec);

					len = parm->dec - strlen(tstr);

					memset(temp, 0, MAX_PARM_LEN);

					for(a = 0; a < len; a++)
						sprintf(temp, "0");

					strgcat(str, temp);

				} else
					strgcat(str, temp);

			}

			return 0;
		}
		case 'D':
		case 'd':                    /* The parameter is a date.   */
		{
			memset(temp, 0, MAX_PARM_LEN);
			memset(str, 0, MAX_PARM_LEN);

			/* Start with separation character. */
			strcopy(str, sepchar);

			/* Get formated date and append it to str. */
			generate_date_string(parm->parmlen, temp);
			strgcat(str, temp);

			return 0;
		}
		default:
			return -1;
	}

} /* end generate_parameter() */


int	generate_request(mptr)
char	*mptr;
/*
	IN:   mptr	A pointer to a message string for the request message.

	OUT:  Returns:	0, if the request was generated properly.
			-1, if the request was not generated properly.

	PURPOSE:  This function generates a request message for the universal
		  driver.  It uses calls to hash_action() and
		  generate_parameter() to generate the request message.
*/
{
VALID_FSERVER	item;
PARM_LIST	*parm;
int		val;
char		*str, dtm[MAX_PARM_LEN], ctm[MAX_PARM_LEN];

	str = (char *) calloc(1, sizeof(char) * MAX_PARM_LEN);

	memset(mptr, 0, MAXMSGLEN);
	memset(str, 0, MAX_PARM_LEN);

	generate_date_string(12, dtm);

	val = wrap_rnd_ri(NUM_DEVICES);

	/* Check whether or not the driver has been waiting for an inquiry
	   server reply.  If it has been, there is a request that has already
	   been partially generated, so the driver can skip ahead and complete
	   the parameter generation for the partially generated request. */
	if (!inquiry_reply_received)
	{
		/* Do the initial request generation.  Start by allocating space
		   for the VALID_FSERVER structure. */
		item.group = (char *) calloc(1, sizeof(char) * _GRP_ID);
		item.sname = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
		item.fname = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
		item.trans = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
		item.num_parms = 0;
		item.fparm = item.lparm = NULL;
		item.next = NULL;

		/* Randomly select a function and function server for use in
		   this request. */
		if (hash_action(&item, HASH_ACTION_RNDSEL))
			return -1;

		/* Check to see if we want to send a bad request for testing
		   purposes. */
		if (send_bad_requests && (!(trans_key % 10)))
		{
			bad_request = 1;

			/* Generate one of three types of bad requests at
			   random.      */
			switch(val % 3)
			{
				/* Send a request with a bad server name. */
				case 0:
				{
					strcopy(item.sname, "bad server name");
					break;
				}

				/* Send a request with a bad funciton name. */
				case 1:
				{
					strcopy(item.fname, "bad function name");
					break;
				}

				/* Send a request with the wrong number of
				   parameters.*/
				case 2:
				{
					if (item.num_parms)
						item.num_parms--;

					break;
				}
			}
		}

		generate_date_string(12, ctm);

		/* Begin filling in the request message. */
		sprintf(mptr,
			"*mpt%d%sDDC_REQUEST%s%d%s%s%s%s%s%d%s%s%s%s%s%s%s%s%s%s",
			val, sepchar, sepchar, transaction_mode, sepchar, ctm,
			sepchar, dtm, sepchar, trans_key, sepchar, sepchar,
			sepchar, sepchar, sepchar, item.sname, sepchar,
			item.fname, sepchar, item.trans);

		trans_key++;

		parm = item.fparm;

		if (parm == NULL)
			strgcat(mptr, ",");
	}

	/* If we were waiting for a reply from an inquiry function server,
	   retrieve the data for the request that was being generated before the
	   inquiry function server request was made. */
	if (inquiry_reply_received)
	{
		strcopy(mptr, holding_str);
		parm = holding_parm;
	}

	wait_for_inquiry = 0;
	inquiry_reply_received = 0;

	/* Generate the parameters for this request. */
	while(parm != NULL)
	{
		memset(str, 0, MAX_PARM_LEN);

		if (generate_parameter(parm, str))
			return -1;

		strgcat(mptr, str);

		/* Check whether or not the driver needs to make an inquiry
		   server request. */
		if (strcmp(parm->iserver, ""))
		{
			memset(holding_str, 0, MAXMSGLEN);

			/* Hold our current position in creating this request. */
			holding_parm = parm->next;
			strcopy(holding_str, mptr);

			memset(mptr, 0, MAXMSGLEN);

			generate_date_string(12, ctm);

			bad_request = 1;

			/* Form the request to the inquiry function server. */
			sprintf(mptr, "*mpt%d%sDDC_REQUEST%s%d%s%s%s%s%s%d%s%s%s%s%s%s%s%s%s%s",
			val, sepchar, sepchar, transaction_mode, sepchar, ctm,
			sepchar, dtm, sepchar, trans_key, sepchar, sepchar,
			sepchar, sepchar, sepchar, parm->iserver, sepchar,
			parm->iname, sepchar, sepchar);

			strgcat(mptr, str);

			/* Set variables so that we know that we are waiting
			   for a response from the inquiry server and note
			   which trans_key is being waited for.              */
			inquiry_trans_key = trans_key;
			wait_for_inquiry = 1;

			/* Get a new trans_key for the inquiry request. */
			trans_key++;

			/* Force exit of while-loop. */
			break;
		}

		parm = parm->next;

	}

	strgcat(mptr, "\012");

	free(str);

	return 0;

} /* end generate_request() */

int	strcopy(strto, strfrom)
char	*strto, *strfrom;
/*
	IN:   strto	A pointer to the string to be copied into.
	      strfrom	A pointer to the string to be copied.

	OUT:  Returns:	0, if the string is copied correctly.
			-1, if an error occured.

	PURPOSE:  This function is used to copy one string into another.  To
		  accomplish this, the function strcopy() function is called,
		  but first, the string to be copied is checked to determine
		  whether or not it is NULL.
*/
{
	if(strfrom != NULL)
		strcpy(strto, strfrom);
	else
		strcpy(strto, "");
	return 0;

} /* end strcopy() */


int	strgcat(strto, strfrom)
char	*strto, *strfrom;
{
	if ((strto == NULL) && (strfrom == NULL))
		return 0;

	if (strfrom == NULL)
		return 0;

	if (strto == NULL)
	{
		strto = "\0";
		strcpy(strto, strfrom);
		return 0;
	}

	strcat(strto, strfrom);

	return 0;

} /* end strgcat() */

int     drv_msgsnd(msnd_id, msnd_buf, msnd_sz, msnd_flg)
int		msnd_id, msnd_sz, msnd_flg;
MSGFORM	*msnd_buf;
{
	int     rtnval;

	fprintf( stderr, "Driver: Send Msg Type     : %ld\n", msnd_buf->mtype );
	fprintf( stderr, "Driver: Send Msg String   : %s\n", msnd_buf->mtext );

	rtnval = MSltSend(msnd_id,
			  msnd_buf->mtype,
			  msnd_buf->mtext,
			  strlen(msnd_buf->mtext) + 1,
			  TRUE );

	if (rtnval == 1)
	{
	  rtnval = 0;
	}
	else
	{
	  rtnval = -1;
	  fprintf(stderr, "Mail slot send error\n");
	}

	return (rtnval);

}

int     srv_msgsnd(msnd_id, msnd_buf, msnd_sz, msnd_flg)
int		msnd_id, msnd_sz, msnd_flg;
MSGFORM	*msnd_buf;
{
	int     rtnval;

	fprintf( stderr, "Server: Send Msg Type     : %ld\n", msnd_buf->mtype );
	fprintf( stderr, "Server: Send Msg String   : %s\n", msnd_buf->mtext );

	rtnval = MSltSend(msnd_id,
			  msnd_buf->mtype,
			  msnd_buf->mtext,
			  strlen(msnd_buf->mtext) + 1,
			  TRUE );
	if (rtnval == 1)
	{
	  rtnval = 0;
	}
	else
	{
	  rtnval = -1;
	}

	return (rtnval);

}

int		drv_msgrcv(mrcv_id, mrcv_buf, mrcv_sz, mrcv_typ, mrcv_flg)
int		mrcv_id, mrcv_sz, mrcv_flg;
long	mrcv_typ;
MSGFORM	*mrcv_buf;
{
	int     rtnval = 0;

	switch( mrcv_flg )
	{
		case DDC_NOWAIT: 
		{
			mrcv_flg = 0;
			break;
		}
		default: 
		{
		 	fprintf(stderr, "Driver:  Wait for Receive.\n");
			mrcv_flg = -1;
			break;
		}
	}

	fprintf( stderr, "Driver: Rcv Msg Type  : %ld\n", mrcv_typ );

	mrcv_buf->mtype = MSltRecv(mrcv_id,
					mrcv_typ,
					mrcv_flg,
					mrcv_buf->mtext,
					&rtnval,
					TRUE );
#ifndef _WIN32

	if (mrcv_buf->mtype==0 && rtnval !=-1)
	{
	  mrcv_buf->mtype=1;
	}

#endif

	fprintf( stderr, "Driver: Rcvd Msg Type  : %ld\n", mrcv_buf->mtype );
	fprintf( stderr, "Driver: Rcvd Msg String: %s\n", mrcv_buf->mtext );

	return (rtnval);
}

int		srv_msgrcv(mrcv_id, mrcv_buf, mrcv_sz, mrcv_typ, mrcv_flg)
int		mrcv_id, mrcv_sz, mrcv_flg;
long	mrcv_typ;
MSGFORM	*mrcv_buf;
{
	int     rtnval = 0;

	switch( mrcv_flg )
	{
		case DDC_NOWAIT: 
		{
			mrcv_flg = 0;
			break;
		}
		default: 
		{
			fprintf(stderr, "Server:  Wait for Receive.\n");
			mrcv_flg = -1;
			break;
		}
	}

	fprintf( stderr, "Server: Rcv Msg Type  : %ld\n", mrcv_typ );

	mrcv_buf->mtype = MSltRecv(mrcv_id,
					mrcv_typ,
					mrcv_flg,
					mrcv_buf->mtext,
					&rtnval,
					TRUE );
#ifndef _WIN32

	if (mrcv_buf->mtype==0 && rtnval !=-1)
	{
	  mrcv_buf->mtype=1;
	}

#endif

	fprintf( stderr, "Server: Rcvd Msg Type  : %ld\n", mrcv_buf->mtype );
	fprintf( stderr, "Server: Rcvd Msg String: %s\n", mrcv_buf->mtext );

	return (rtnval);
}

long	wrap_rnd_ri(long val)
{
	return(rnd_ri(val));
} /* end wrap_rnd_ri() */
