
/***********************************************************************
* File: ddc_driver.c  (Test version, real version = ddc_driver.orig)
* 
* This file contains the main program (handler) which is implemented
* by the DDC Device driver supplier
* 
***********************************************************************/
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#ifdef _WIN32
# include <windows.h>
# include <process.h>
# include <io.h>
#else	/* _WIN32 */
# include <sys/ipc.h>
#endif

#include "system.h"
#include "ddc_baan.h"
#include "ddc.h"
#include "ddc_supplier.h"
#include "ddc_tools.h"

#ifdef __STDC__
#define CONST	const
#endif

#ifdef FILELINE
#undef FILELINE
#endif
#define FILELINE	fprintf(stderr, "File:  '%s' line:  %d\n", __FILE__, __LINE__);

ACTIVE_LVF      **lvflist;
int             num_requests;
int             available_trans;
int             available_lvf;
int             ddc_initiate;
char    **reqlist;
char    *dummy_ptr;
LVF_REPEAT_FORM *lvf_rep;

#ifdef  _WIN32
__declspec (dllimport) int              readreq;
__declspec (dllimport) int              open_trans;
__declspec (dllimport) int              open_lvf;
__declspec (dllimport) int              cur_requests;
__declspec (dllimport) int              ddc_logging;
__declspec (dllimport) int              logvar;
__declspec (dllimport) FILE             *debug_fd;
__declspec (dllimport) int              mode;
__declspec (dllimport) int              transaction_mode;
__declspec (dllimport) int              wait_for_inquiry;
__declspec (dllimport) int              inquiry_reply_received;
__declspec (dllimport) char             sepchar[2];
__declspec (dllimport) int              lastread;
__declspec (dllimport) int              num_infiles;
__declspec (dllimport) FILE             *infile[MAX_INPUT_FILES];
__declspec (dllimport) int              allowed_msglen;
__declspec (dllimport) char             dummy[256];
__declspec (dllimport) char             *bse_dir;
__declspec (dllimport) int              msgid_in;
__declspec (dllimport) int              msgid_out;
__declspec (dllimport) int              msgid_LVF;
#else
int     readreq;
int     open_trans;
int     open_lvf;
int     cur_requests;
int     ddc_logging;
int           logvar;
FILE    *debug_fd;
int             mode;
int             transaction_mode;
int             wait_for_inquiry;
int             inquiry_reply_received;
char    sepchar[2];
int             lastread;
int             num_infiles;
FILE    *infile[MAX_INPUT_FILES];
int             allowed_msglen;
char    dummy[256];
char    *bse_dir;
int              msgin_in;
int              msgin_out;
int              msgin_LVF;
#endif

int		copy_LVF_header (LVF_HEADER *, LVF_HEADER *);
int		copy_LVF_element (char *, char *, char *, LVF_FORM *);
void		process (void);
int		create_term_list (TERMINALS **);
int		send_test_request (REQUEST_FORM*);
SignalType	clean_kill (int);
int		print_menu_tree (GROUPLIST *);
int		print_menu (MENU_HEADER *, char *, int);
int		print_function (FORM_HEADER *, char *, int);
int		send_repeat_message (char *);
int		lvf_action (char*, int);
int		transaction_action (char*, int);
int		send_request (REQUEST_FORM*, char*);
int		send_end (void);
void		shutdown_drv (char*, char*, char*, int, int*, int, 
		REQUEST_FORM*,REPLY_FORM*, LVF_HEADER*, LVF_HEADER*);

char		driver_name[256];

char *Version = "@(#)(B2 ) Version     : ddcdriver%s version 01-1994/R1";

int	send_repeat_message( message )
char 	*message;
/*
	IN:   message	A message line that was recieved from the server.

	OUT:  returns:  0, if message was sent properly.
			-1, if an error occured.

	PURPOSE:  This function sends a message to the server, requesting that
		  it repeats a message, due to a problem with the data that
		  was recieved.
*/
{
int	fnum, len;
char 	buff[MAXMSGLEN], *field;

	if (!message)
	   return -1;

	lvf_rep->dtype = 0;
	strcopy(lvf_rep->ddc_id, "");
	strcopy(lvf_rep->LVFname, ""); 
	strcopy(lvf_rep->dtime, ""); 
	for (fnum=1; fnum < LSTR_NUM_PARMS + 1; fnum++)
	{
	    	memset(buff, 0, MAXMSGLEN);
	    	len = get_field(&message, buff);  /* Get next message field. */
		field = (&(buff[0]));

	    	if (len <=0)
			continue;

		switch (fnum)           /* Choose the field to be filled in. */
		{
			case LSTR_ID:
				strcopy(lvf_rep->ddc_id, buff);
				break;
			case LSTR_LS:
				break;
			case LSTR_LI:
				strcopy(lvf_rep->LVFname, buff);
				break;
			case LSTR_DO:
				lvf_rep->dtype = atoi(field);
				break;
			case LSTR_DT:
				strcopy(lvf_rep->dtime, buff);
				break;
		}
	}

	ddc_send_LVF_repeat(lvf_rep);  /* Send the message to the server.  */
	show_lvf_repeat(lvf_rep);      /* Show what was done for debugging */
                                       /* purposes.                        */
	return 0;

} /* end send_repeat_message() */


int 	print_function( cfunc, groupid, level)
FORM_HEADER	*cfunc;
char		*groupid;
int		level;
/*
	IN:   cfunc	The address of a the form header of a function.
	      groupid	The group that the function is a part of.
	      level	The current level in the menu structure.

	OUT:  returns:	0, if the function was displayed properly.
			-1, if an error occured.

	PURPOSE:  This function is used to display a function and its
		  information.  This function facilitates the displaying
		  of a function by determining what information should be
		  displayed in what order, and calling mesg_print().  It
		  also fills in a VALID_FSERVER structure for each function
		  that is printed and calls hash_action() to add the
		  VALID_FSERVER structure to the list of valid function
		  servers.
*/
{
int i;
char	*temp;		/* Used to handle ptype in the FORM_FIELD   */
			/* structure.  This is necessary, because   */
			/* ptype is a character, not a string, and  */
			/* mesg_print() looks for strings as input. */
char 	*spacestr;
VALID_FSERVER	item;
FORM_FIELD	*cfline;
COMB_FIELD	*ccfline;
PARM_LIST	*parm;

	if (!cfunc)
	     return -1;

	spacestr = (char *) calloc (1, sizeof(char) * level + 2);

	for (i=0; i < level; i++)
	{
		strgcat(spacestr, "   ");
	}

	item.group = (char *) calloc(1, sizeof(char) * _GRP_ID);
	item.sname = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
	item.fname = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
	item.trans = (char *) calloc(1, sizeof(char) * MAXNAMELEN);

	strcopy(item.group, groupid);
	strcopy(item.sname, cfunc->fserv);
	strcopy(item.fname, cfunc->fname);

	if (cfunc->trans_pre)
		strcopy(item.trans, cfunc->trans_pre);

	item.num_parms = cfunc->nrpars;
	item.fparm = item.lparm = NULL;
	item.next = NULL;

	mesg_print(spacestr, DBG_MSG_1, cfunc->fserv, "", 0, 0,
		   MSG_FORMAT_1, DRIVER_REC);

	mesg_print(spacestr, DBG_MSG_2, cfunc->fname, "", 0, 0,
		   MSG_FORMAT_1, DRIVER_REC);

	mesg_print(spacestr, DBG_MSG_3, "", "", cfunc->update_mode,
		   0, MSG_FORMAT_2, DRIVER_REC);

	mesg_print(spacestr, DBG_MSG_4, cfunc->trans_pre, "", 0, 0,
		   MSG_FORMAT_1, DRIVER_REC);

	cfline = cfunc->fline;
	ccfline = cfunc->cfield;

	temp = (char*) calloc (1, sizeof(char));

	for (i=0; i < cfunc->nrpars; i++) 
	{
		parm = (PARM_LIST *) calloc(1, sizeof(PARM_LIST));
		initialize_parameter_structure(parm);

		if(cfline->prompt == NULL)
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_6, "",
			   	   i+1, 0, MSG_FORMAT_3, DRIVER_REC);
		else
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_6, 
				   cfline->prompt, i+1, 0, MSG_FORMAT_3,
				   DRIVER_REC);

		(*temp) = cfline->ptype;
		if(temp == NULL)
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_7, "", i+1,
				   0, MSG_FORMAT_3, DRIVER_REC);
		else
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_7, temp, i+1,
				   0, MSG_FORMAT_3, DRIVER_REC);

		if(cfline->pvalue == NULL)
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_8, "",
				   i+1, 0, MSG_FORMAT_3, DRIVER_REC);
		else
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_8,
				   cfline->pvalue, i+1, 0, MSG_FORMAT_3, 
				   DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_5, DBG_MSG_9, "", i+1,
			   cfline->pseqnr, MSG_FORMAT_4, DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_5, DBG_MSG_10, "", i+1,
			   cfline->plen, MSG_FORMAT_4, DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_5, DBG_MSG_11, "", i+1,
			   cfline->pdlen, MSG_FORMAT_4, DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_5, DBG_MSG_12, "", i+1,
			   cfline->justify, MSG_FORMAT_4, DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_5, DBG_MSG_13, "", i+1,
			   cfline->zero_filled, MSG_FORMAT_4, DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_5, DBG_MSG_14, "", i+1,
			   cfline->par_vtype, MSG_FORMAT_4, DRIVER_REC);

		if(cfline->listid == NULL)
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_15, "",
			   	   i+1, 0, MSG_FORMAT_3, DRIVER_REC);
		else
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_15,
				   cfline->listid, i+1, 0, MSG_FORMAT_3,
				   DRIVER_REC);

		if(cfline->err_msg == NULL)
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_16, "",
			   i+1, 0, MSG_FORMAT_3, DRIVER_REC);
		else
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_16, 
				   cfline->err_msg, i+1, 0, MSG_FORMAT_3, 
				   DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_5, DBG_MSG_17, "", i+1,
			   cfline->one_time_f, MSG_FORMAT_4, DRIVER_REC);

		if(cfline->field_pre == NULL)
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_18, "",
			   	   i+1, 0, MSG_FORMAT_3, DRIVER_REC);
		else
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_18, 
				   cfline->field_pre, i+1, 0, MSG_FORMAT_3, 
				   DRIVER_REC);

		if(cfline->bitmask == NULL)
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_19, "",
			   	   i+1, 0, MSG_FORMAT_3, DRIVER_REC);
		else
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_19, 
				   cfline->bitmask, i+1, 0, MSG_FORMAT_3, 
				   DRIVER_REC);

		if(cfline->template == NULL)
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_20, "",
			   	   i+1, 0, MSG_FORMAT_3, DRIVER_REC);
		else
			mesg_print(spacestr, DBG_MSG_5, DBG_MSG_20, 
				   cfline->template, i+1, 0, MSG_FORMAT_3, 
				   DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_5, DBG_MSG_21, "", i+1,
			   cfline->fe_timeout, MSG_FORMAT_4, DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_5, DBG_MSG_22, "", i+1,
			   cfline->max_retry, MSG_FORMAT_4, DRIVER_REC);

		parm->ptype = cfline->ptype;
		if (cfline->pvalue)
			strcopy(parm->pvalue, cfline->pvalue);
		parm->pseqnr = cfline->pseqnr;
		parm->parmlen = cfline->plen;
		parm->dec = cfline->pdlen;
		parm->just = cfline->justify;
		parm->zfill = cfline->zero_filled;
		parm->parmtype = cfline->par_vtype;
		parm->onetime = cfline->one_time_f;
		if (cfline->listid)
			strcopy(parm->listid, cfline->listid);
		if (cfline->bitmask)
			strcopy(parm->bitmask, cfline->bitmask);
		if (cfline->template)
			strcopy(parm->template, cfline->template);
		if (cfline->inqfserv)
			strcopy(parm->iserver, cfline->inqfserv);
		if (cfline->inqfname)
			strcopy(parm->iname, cfline->inqfname);

		/* Add first parameter to the list. */
		if (item.fparm == NULL)	
			item.fparm = item.lparm = parm;

		/* Add second parameter to the list. */
		else if (item.fparm == item.lparm)
			item.fparm->next = item.lparm = parm;

		/* Add additional parameter to the list. */
		else
		{
			item.lparm->next = parm;
			item.lparm = parm;
		}

		cfline = cfline->lnext;
		memset(temp, 0, 1);
	}

	free(temp);

	if (hash_action(&item, HASH_ACTION_ADD))
	{
		mesg_print(spacestr, "Server already exists ", "\n", "",
			   0, 0, MSG_FORMAT_1, DRIVER_ERR);
	}

	while (ccfline)
	{
		mesg_print(spacestr, DBG_MSG_23, ccfline->field_id, "", 0, 0,
			   MSG_FORMAT_1, DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_24, ccfline->trans_prefix, "",
			   0, 0, MSG_FORMAT_1, DRIVER_REC);

		mesg_print(spacestr, DBG_MSG_25, "", "", ccfline->seqno, 0,
			   MSG_FORMAT_2, DRIVER_REC);

		ccfline = ccfline->lnext;
	}

	memset(item.group, 0, _GRP_ID);
	memset(item.sname, 0, MAXNAMELEN);
	memset(item.fname, 0, MAXNAMELEN);
	memset(item.trans, 0, MAXNAMELEN);

	free(spacestr);
	free(item.group);
	free(item.sname);
	free(item.fname);
	free(item.trans);

	return 0;

} /* end print_function() */


int 	print_menu( cmenu, groupid, level )
MENU_HEADER	*cmenu;
char		*groupid;
int		level;
/*
	IN:   cmenu	A pointer to the menu header of a menu.
	      groupid	The group for which the current menu is being printed.
	      level	The level of the current menu in the menu structure.

	OUT:  returns:	0, if the menu was displayed correctly.
			-1, if an error occured.

	PURPOSE:  This function displays a menu that is given by the caller.
		  When called, this function loops through each menu line of
		  the given menu and determines whether or not the line
		  refers to a submenu or a function.

		  If a menu line refers to a submenu, this function uses calls
		  to mesg_print() to display the identification and description
		  of the menu.  After the calls to mesg_print, this function
		  makes a recursive call to itself, passing the menu header of
		  the current menu as an argument, thus allowing the entire
		  menu structure to be displayed.

		  If a menu line refers to a function, this function calls
		  print_function(), passing the function header as an
		  argument.
*/
{
MENU_LINE	*cline;
int 	i, nlevel;
char 	*spacestr;

	spacestr = (char *) calloc (1, sizeof(char) * level + 2);

	for (i=0; i < level; i++)
	{
		strgcat(spacestr, "   ");
	}

	nlevel = level + 1;
	for (i=1; i <= cmenu->nrlines; i++)
	{
		cline = ddc_get_menu_line(cmenu, i);
		if (ddc_is_line_a_function(cline))
		{
			print_function( cline->fhead, groupid, level);
		}
		else
		{
			mesg_print(spacestr, DBG_MSG_26, cline->ident, "",
				   0, 0, MSG_FORMAT_1, DRIVER_REC);

			mesg_print(spacestr, DBG_MSG_27, cline->description,
				   "", 0, 0, MSG_FORMAT_1, DRIVER_REC);

			if (cline->fhead || cline->mhead)
				print_menu( cline->mhead, groupid, nlevel);
			else
			mesg_print(spacestr, DBG_MSG_29, "", "", 0, 0,
					   MSG_FORMAT_5, DRIVER_ERR);
		}
	}
	free(spacestr);

	return 0;

} /* end print_menu() */


int 	print_menu_tree( glist )
GROUPLIST       *glist;
/*
	IN:   glist	A pointer to the list of selected groups.

	OUT:  returns:	0, if entire menu tree printed correctly.
			-1, if an error occured.

	PURPOSE:  This function displays the entire menu tree for each group
		  in the selected group list.  It loops through the group
		  list, and for each group, it displays the group
		  identification and calls print_menu(), to display the group's
		  menu structure.
*/
{
GROUPLIST	*tlist;
MENU_HEADER	*menumain;

	tlist = glist;
	while ( tlist )
	{
		menumain = ddc_get_mainmenu_entry (tlist->grpident);
		
		if (menumain)
		{
			mesg_print(DBG_MSG_28, tlist->grpident, "", "", 0, 0, MSG_FORMAT_5, DRIVER_REC);
			print_menu( menumain, tlist->grpident, 0 );
		}
		tlist = tlist->lnext;
	}

	return 0;
} /* end print_menu_tree() */


int	lvf_action(lvfname, action)
char	*lvfname;
int	action;
/*
	IN:   lvfname	A pointer to a string containing a lvf name.
	      action	An action code:
			  LVF_ACTION_NUMRECS	Return number of records
						for the requested lvf.
			  LVF_ACTION_ADDELEM	Add an element to the
						requested lvf.
			  LVF_ACTION_ADDLVF	Add a new lvf to lvflist.
			  LVF_ACTION_REMLVF	Remove requested lvf from
						lvflist.

	OUT:  Output for this function varies depending on which action
	      is performed.
			LVF_ACTION_NUMRECS recs	The number of records in
						the requested lvf.
					   -1	Requested lvf not found.
			LVF_ACTION_ADDELEM 0	Element added successfully.
					   -1	Requested lvf not found.
			LVF_ACTION_ADDLVF  0	LVF added successfully.
					   -1	LVF could not be added.
			LVF_ACTION_REMLVF  0	LVF successfully removed.
					   -1	Requested lvf not found.

	PURPOSE:  This function performs an action on an lvf.  The caller
		  gives as input an lvf name and an action code, and this
		  function performs the specified action to the given lvf.
*/
{
int	i = 0;

	if (action == LVF_ACTION_ADDLVF)
	{
		if (open_lvf == MAX_LVFS)
			return -1;

		if (open_lvf >= available_lvf)
		{
			available_lvf = available_lvf + NUM_INIT_LVFS;
			lvflist = (ACTIVE_LVF **) realloc( lvflist,
			available_lvf * sizeof (ACTIVE_LVF *));


			for(i = open_lvf; i < available_lvf; i++)
			{
				lvflist[i] = (ACTIVE_LVF *) calloc(1,
						sizeof(ACTIVE_LVF));
				lvflist[i]->lvfname = (char *) calloc (1,
							sizeof(char) * _LISTID);
				lvflist[i]->recs = 0;
			}
		}

		i = open_lvf;
	} else {
		while(strcmp(lvfname, lvflist[i]->lvfname) && i < open_lvf)
			i++;

		if (i >= open_lvf)
			return -1;
	}

	switch(action)
	{
		case LVF_ACTION_NUMRECS:
		{
			return (lvflist[i]->recs);
		}

		case LVF_ACTION_ADDELEM:
		{
			lvflist[i]->recs = lvflist[i]->recs + 1;
			return 0;
		}

		case LVF_ACTION_ADDLVF:
		{
			strcopy(lvflist[i]->lvfname, lvfname);
			lvflist[i]->recs = 0;
			open_lvf++;

			return 0;
		}

		case LVF_ACTION_REMLVF:
		{
			while(i < (open_lvf - 1))
			{
				strcopy(lvflist[i]->lvfname,
				       lvflist[i+1]->lvfname);

				lvflist[i]->recs = lvflist[i+1]->recs;

				i++;
			}

			strcopy(lvflist[i]->lvfname, "");
			lvflist[i]->recs = 0;
			open_lvf--;

			return 0;
		}
		default:
			return -1;
	}

} /* end lvf_action() */


int	transaction_action(transptr, action)
char	*transptr;
int	action;
/*
	IN:   transptr	A pointer to a string containing a transaction name.
	      action	An action code:
			  TRAN_ACTION_REMOVE	Remove a transaction from reqlist.
			  TRAN_ACTION_ADD	Add a transaction to reqlist.
			  TRAN_ACTION_LOOKUP	Look for a transaction in reqlist.

	OUT:  Output for this function varies depending on which action is
	      performed.
			TRAN_ACTION_REMOVE  0	Transaction successfully removed.
					    -1	Requested transaction not found.
			TRAN_ACTION_ADD     0	Transaction successfully added.
					    -1	Transaction could not be added.
			TRAN_ACTION_LOOKUP  0	Transaction was found in reqlist.
					    -1	Transaction was not found.

	PURPOSE:  This function performs an action on a transaction.  The caller
		  gives as input a transaction name and an action code, and this
		  function performs the specified action on the given transaction.
*/
{
int	i = 0;

	switch(action)
	{
		case TRAN_ACTION_REMOVE:
		{
			while (strcmp(transptr, reqlist[i]) && i < open_trans)
				i++;

			if (i >= open_trans)
				return -1;

			while(i < (open_trans - 1))
			{
				strcopy(reqlist[i], reqlist[i+1]);
				i++;
			}

			strcopy(reqlist[i], "");

			open_trans--;
			break;
		}
		case TRAN_ACTION_ADD:
		{
			if (open_trans == MAX_TRANS)
				return -1;

			if (open_trans >= available_trans)
			{
				available_trans = available_trans +
							NUM_INIT_TRANS;
				reqlist = (char **) realloc( reqlist,
					  available_trans * sizeof (char *));
				for(i = open_trans; i < available_trans; i++)
				{
					reqlist[i] = (char *) calloc (1,
							sizeof(char) * 10);

					strcopy(reqlist[i], "");
				}
			}

			strcopy(reqlist[open_trans], transptr);
			open_trans++;
			break;
		}
		case TRAN_ACTION_LOOKUP:
		{
			while (strcmp(transptr, reqlist[i]) && i < open_trans)
				i++;

			if (i >= open_trans)
				return -1;
			else
				return 0;
			
			break;
		}
		default:
			return -1;
	}

	return 0;

} /* end transaction_action() */


SignalType	clean_kill(int signo)
/*
	PURPOSE:  This function is exicuted when the driver wants to stop
		  communications with the server.  It tells the server that
		  the driver is ready to stop, closes things down on the
		  driver side, and exits.
*/
{
    	ddc_stop_server();
	ddc_ready_to_stop();
	exit(0);

} /* end clean_kill() */


int 	copy_LVF_header( source, dest )
LVF_HEADER *source, *dest;
/*
	IN:   source	A pointer to the source lvf header.
	      dest	A pointer to the destination lvf header.

	OUT:  returns:	0, if the fields of source were copied into the fields
			   of dest.
			-1, if an error occured.

	PURPOSE:  This function simply copies all of the information in the
		  fields of source into the fields of dest, thus making an
		  exact copy of source in dest.
*/
{
	if ((!source) || (!dest))
		return -1;

	strcopy(dest->LVFname, source->LVFname);
	strcopy(dest->stime, source->stime);
	dest->listnum = source->listnum;
	dest->storage_loc = source->storage_loc;
	
	return 0;

} /* end copy_LVF_header() */


int	send_request(request, mptr)
REQUEST_FORM *request;
char	*mptr;
/*
	IN:   request	A pointer to a request form.
	      mptr	A pointer to the request message.

	OUT:  returns:	0, if the request was sent properly.
			-2, if an error occured.

	PURPOSE:  This function generates a request form and then calls
		  ddc_send_request().  The string pointed to by mptr is
		  treated as a request message, and the request form is
		  filled in using the data in the request message fields.
		  Once the request form is filled in, the function uses
		  a call to hash_action() to check whether or not the
		  function server which was given is valid.  If it is
		  valid, the transaction is added to the list, and the
		  message is sent.  If it is not valid, the function
		  calls write_mesg() with a message that says that the
		  function server wasn't valid and then returns, without
		  actually sending the message to the server.
*/
{
VALID_FSERVER	item;
char	buff[MAXMSGLEN];
int	len, fnum, i;
struct tm;

	item.group = (char *) calloc(1, sizeof(char) * _GRP_ID);
	item.sname = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
	item.fname = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
	item.trans = (char *) calloc(1, sizeof(char) * MAXNAMELEN);
	item.num_parms = 0;
	item.next = NULL;

	clear_request(&request);

	for (fnum = 1; fnum < REQU_NUM_PARMS; fnum++)
	{
	    memset(buff, 0, MAXMSGLEN); 
	    len = get_field(&mptr, buff);

	    if (len <= 0)
		continue;
			
  	    switch (fnum)
	    {
		case REQU_ID:
		{
#ifdef _WIN32
			//request->ddc_id = (char *) calloc(1, sizeof(ID_LEN));
			memset(request->ddc_id, 0, sizeof(request->ddc_id));
#else		
			strcopy(request->ddc_id, buff);
#endif
			break;
		}
		case REQU_RE:
			break;
		case REQU_RS:
		{
			request->state = atoi(buff);
			break;
		}
		case REQU_DC:
		{
			strcopy(request->device_date, buff); 
			break;
		}
		case REQU_DT:
		{
			strcopy(request->contr_date, buff); 
			break;
		}
		case REQU_TK:
		{
			strcopy(request->trans_key, buff); 
			break;
		}
		case REQU_TV:
		{
			strcopy(request->trans_value, buff); 
			break;
		}
		case REQU_R1:
		{
			strcopy(request->reserved1, buff); 
			break;
		}
		case REQU_R2: 
		{
			strcopy(request->reserved2, buff); 
			break;
		}
		case REQU_R3:
		{
			strcopy(request->reserved3, buff); 
			break;
		}
		case REQU_DM:
		{
			strcopy(request->dname, buff); 
			strcopy(item.sname, buff);
			break;
		}
		case REQU_FS:
		{
			strcopy(request->fname, buff); 
			strcopy(item.fname, buff);
			break;
		}
		case REQU_FN:
		{
			strcopy(request->trans_prefix, buff); 
			strcopy(item.trans, buff);
			break;
		}
	    }
	}

	for (fnum = 0; fnum < MAX_FUN_PARMS; fnum++)
	{
	    	len = get_field(&mptr, buff); 
		i = (int) buff[len-1];

	    	if (i == 10)
	    	{
			request->nr_of_fpars = fnum+1;
			strncpy(request->fpars[fnum], buff,
					len-1);
			fnum = MAX_FUN_PARMS;
			continue;
		}

		strcopy(request->fpars[fnum], buff);
	}

	item.num_parms = request->nr_of_fpars;


	if (hash_action(&item, HASH_ACTION_LOOKUP))
	{
		memset(buff, 0, MAXMSGLEN);
		sprintf(buff, "Function server not valid:  %s, %s, %s, %d.\n",
			item.sname, item.fname, item.trans, item.num_parms);
		write_mesg(DRIVER_ERR, buff);

		free(item.group);
		free(item.sname);
		free(item.fname);
		free(item.trans);

		return 0;
	} else {
		memset(buff, 0, MAXMSGLEN);

		sprintf(buff, "Function server is valid:  %s, %s, %s, %d.\n",
			item.sname, item.fname, item.trans, item.num_parms);
		write_mesg(DRIVER_ERR, buff);
	}

	free(item.group);
	free(item.sname);
	free(item.fname);
	free(item.trans);

	if (open_trans > MAX_TRANS)
		return 0;

	transaction_action(request->trans_key, TRAN_ACTION_ADD);

	if (!ddc_send_request(request))
		raise(SIGTERM);

	show_request_data(request);

	return 0;

} /* end send_request() */


int	send_end()
/*
	PURPOSE:  This function simply calls ddc_stop_server() and changes
		  mode to 0.  This function is called by process, whenever
		  there is a request to end communications with the server.
*/
{
	ddc_stop_server();
	mode = 0;

	return 0;

} /* end send_end() */


int 	send_test_request(request)
REQUEST_FORM  *request;
/*
	IN:   request	A request form.

	OUT:  returns	retval  lf the request was sent properly.
			-1      If an error occured.
			-2	If the request is DDC_END or DDC_END_REQ.

	PURPOSE:  This function handles a test request.  First, it checks
		  whether or not an input file is being used for requests.
		  If there is an input file, this function reads the next line
		  of the file into inmesg.  Then, it uses mptr as an index to
		  scan the fields of the message to determine what kind of
		  request it is.  If the request is DDC_END or DDC_END_REQ,
		  -2 is returned.  If the request is DDC_LIST_REPEAT,
		  send_repeat_message() is called.  If the request is anything
		  else a request from, send_request() is called.  If there are
		  no input files left to read from, -2 is returned.  If no
		  input files are being used, this function calls
		  generate_request() to get a request message and then calls
		  send_request() until the number of requests to be sent is
		  reached.  Once all of the requests that are supposed to be
		  sent have been sent, this function returns -2.
*/
{
char	*mptr, inmesg[MAXMSGLEN], buff[MAXMSGLEN], *field, *newmsg;
int		len, i, temp, retval;
FILE	*infd;

	memset(inmesg, 0, MAXMSGLEN); 
	mptr = (&(inmesg[0]));
	field = (&(buff[0]));

	if (!readreq)
	{
		/* Input files are to be used to get requests to be sent. */
		which_input_file(&infd);  /* this function will search for the input file to read from */
		
		if( fgets(mptr, MAXMSGLEN, infd) == NULL)         
			fprintf(stderr, "fgets error\n" );
		else         
			fprintf(stderr, "Driver: reads this request from input file: %s \n\n", mptr);      
		
		i = 1;
		temp = lastread;

		/* If the input file is empty, find the next input file that
		   still contains requests.  If all of the files are empty,
		   return -2.                                                */

		while(!strlen(mptr))
		{
			if (i >= num_infiles)
				return -2;

			temp++;

			if (temp < num_infiles)
				fgets(mptr, MAXMSGLEN, infile[temp]);
			else {
				temp = 0;
				fgets(mptr, MAXMSGLEN, infile[temp]);
			}

			i++;
		}

		lastread = temp + 1;

	} else {
		/* The request generator is to be used to generate and send
		   num_requests requests.                                   */

		if (!inquiry_reply_received)
			cur_requests++;

		if (cur_requests <= num_requests)
		{
			newmsg = (char *) calloc(1, sizeof(char) * MAXMSGLEN);

			generate_request(newmsg);
			strcopy(mptr, newmsg);

			free(newmsg);
		} else
			return -2;
	}

	get_field(&mptr, buff);       /* Get second field of message  */
    	memset(buff, 0, MAXMSGLEN);   /* because it contains the type */
	len = get_field(&mptr, buff); /* code for the message.        */

    	if (len <=0)
		return -2;

	if (!strncmp("DDC_END", buff, strlen("DDC_END")) ||
	    !strncmp("DDC_END_REQ", buff, strlen("DDC_END_REQ")))
		return -2;
	else
		if (!strncmp("DDC_LIST_REPEAT", buff,
	   	   strlen("DDC_LIST_REPEAT")))
		{
			mptr = (&(inmesg[0]));
			send_repeat_message(mptr);
			return 1;
		}

	mptr = (&(inmesg[0]));
fprintf(stderr, "mptr = %s\n\n", mptr);
	retval = send_request(request, mptr);
	return retval;

} /* end send_test_request() */


int	create_term_list( term )
TERMINALS **term;
/*
	IN:   term	A pointer to a pointer of type TERMINAL.

	OUT:  returns	0, if the terminal list was generated properly.
			-1, if an error occured.

	PURPOSE:  This function creates a list of terminals.  First, it
		  allocates space for a terminal.  Then, it assigns the new
		  terminal a generic id and attaches it to the terminal list.
*/
{
	register int i;
	TERMINALS	*tmp;

	for (i=9; i>0; i--){
		tmp = (TERMINALS *) calloc (1, sizeof(TERMINALS));

		if (i < 10)
			sprintf( tmp->term_id, "*TERM_00%d", i);
		else {
			if (i < 100)
				sprintf( tmp->term_id, "*TERM_0%d", i);
			else
				sprintf( tmp->term_id, "*TERM_%d", i);
		}

		tmp->lnext = *term;
		*term = tmp;	
	}
	return ( 0 );

} /* end create_term_list() */


int 	copy_LVF_element ( rec, actn, list, elem )
char		*rec, *actn, *list;
LVF_FORM 	*elem;
/*
	IN:   rec	A character string.
	      actn	A character string.
	      list	A character string.
	      elem	A pointer to an LVF_FORM.

	OUT:  actn	A copy of elem's action field.
	      rec	A copy of elem's record field.
	      list	A copy of elem's listid field.
	      returns	0, if the desired fields are copied correctly.
			-1, if an error occurs.

	PURPOSE:  This function copies certain fields of an element.  For each
		  of the three fields, the corresponding string pointer is
		  checked to see if it actually points to a memory location.
		  For every string pointer that points to valid memory, the
		  corresponding element field is copied.
*/
{
	if ( actn ) strcopy(elem->action, actn);
	if ( rec ) strcopy(elem->record, rec);
	if ( list ) strcopy(elem->listid, list);

	return 0;

} /* end copy_LVF_element() */


void	shutdown_drv(LVFrecord, LVFname, action, max_size, cur_size, repret,
			 request, reply, lvf, lvf2)
char			*LVFrecord, *LVFname, *action;
int				max_size, *cur_size, repret;
REQUEST_FORM	*request;
REPLY_FORM		*reply;
LVF_HEADER		*lvf, *lvf2;
/*
	OUT:   Returns:	0, if the queues are cleaned up without problems.
			-1, if an error occured.

	PURPOSE:  This function cleans up any messages remaining in the message
		  queues.  It takes as input all of the variables used in the
		  calls to ddc_get_LVFrecord() and ddc_get_reply().  If there
		  are no problems, this function returns a one, otherwise, it
		  returns a zero.
*/
{
	memset(LVFrecord, 0, max_size);
	strcopy(LVFrecord, "CLEANING UP MESSAGES LEFT IN THE QUEUES\n\n");
	write_mesg(DRIVER_SENT, LVFrecord);
	
	/* Clean up any messages in LVF queue. */
	repret = 0;
	while (repret != -1 && ddc_get_LVFs_pending())
	{
		repret = ddc_get_LVFrecord( LVFname, LVFrecord, max_size,
			cur_size, action );
	}

	/* Clean up any messages in the in queue. */
	repret = 1;
	while (repret > 0)
	{
		repret = ddc_get_reply(&reply, &lvf, &lvf2, DDC_NOWAIT);

		if (repret == 1 || repret == 2)
		{
			show_reply_data(reply);
		}
	}

} /* end shutdown_drv() */


void 	process()
/*
	IN:   none

	OUT:  none

	PURPOSE:  This function serves as the engine for the universal driver.
		  It exicutes in three stages, initialization, exicution, and
		  clean-up.

		  In the initialization stage, three things are done.  First,
		  memory is allocated for the pointer variables.  Second, this
		  function calls the initialize functions to initialize all of
		  the global variables to be used.  Finally, it allocates
		  memory for both the request and lvf lists.

		  In the exicution stage, this function sends the first request
		  to the server and performs a loop to handle events until it
		  is time to stop.  In the loop, the driver first calls
		  ddc_get_reply() to get a reply off the message queue, if one
		  exists.  Then, it handles replies or LVFs, based on the return
		  value of ddc_get_reply().  Next, it checks the LVF message
		  queue and handles any incoming LVFs.  Finally, it calls
		  send_test_request() to send the next request to the server and
		  continues the loop until either the server sends a DDC_END_REQ
		  message or send_test_request() returns a value of -2.

		  In the clean-up stage, the universal driver prepares to shut
		  down.  First, the procedure shutdown_drv() is called to clean up
		  any messages remaining in the message queues.  Then, this
		  function frees the memory that the reply and LVF lists are
		  using.  Next, it calls hash_action() to free the memory being
		  used by the valid function server table.  Then, it frees the
		  memory that was allocated for the pointer variables.  Finally,
		  it calls ddc_ready_to_stop() to inform the server that it is
		  ready to stop and returns.
*/
{
REQUEST_FORM	*request;
REPLY_FORM	*reply, *rep = ((REPLY_FORM *) 0);
LVF_HEADER 	*lvftmp, *lvfprev, *lvfhead, *lvf, *lvf2;
LVF_FORM 	*lvf_rec;
LVF_REPEAT_FORM	*lvf_rep;
VALID_FSERVER   item;
char 		*LVFname = ((char *) 0), *LVFrecord = ((char *) 0), *action;
char		temp[100];
int		max_size = 128, *cur_size, i, repret, lvfopen, 
		retval, nomesg;

/* INITIALIZATION STAGE */

	lvftmp = lvfprev = lvfhead = ((LVF_HEADER *) 0);
	nomesg = 0;

	/* Allocate space for pointer variables. */
	cur_size = (int *) calloc (1, sizeof (int));
	LVFrecord = (char *) calloc (1,
			sizeof (char) * max_size); 
	LVFname = (char *) calloc (1, sizeof
		(char) * 25);
	action = (char *) calloc (1, sizeof (char) ); 

	/* Call functions to initialize the global variables. */
	initialize_request(&request);
	clear_request(&request);
	initialize_reply(&reply);
	initialize_lvf(&lvf);
	initialize_lvf(&lvf2);
	initialize_lvf_repeat(&lvf_rep);
	initialize_LVF_element(&lvf_rec, max_size);


	/* Allocate space for the request list.  The list is initially
	   allocated to track a number of requests specified by
	   NUM_INIT_TRANS.  The list can be expanded to track more
	   requests using realloc.                                      */
	reqlist = (char **) calloc(NUM_INIT_TRANS, sizeof(char *));
	for (i = 0; i < NUM_INIT_TRANS; i++)
	{
		reqlist[i] = (char *) calloc (1, sizeof(char) * 10);
		strcopy(reqlist[i],"");
	}

	available_trans = NUM_INIT_TRANS;

	/* Allocate space for the lvf list.  The list is initially
	   allocated to track a number of lvfs specified by
	   NUM_INIT_LVFS.  The list can be expanded to track more lvfs
	   using realloc.						*/
	lvflist = (ACTIVE_LVF **) calloc(NUM_INIT_LVFS, sizeof(ACTIVE_LVF *));
	for (i = 0; i < NUM_INIT_LVFS; i++)
	{
		lvflist[i]= (ACTIVE_LVF *) calloc(1, sizeof(ACTIVE_LVF));
		lvflist[i]->lvfname = (char *) calloc
						(1, sizeof(char)* _LISTID);
		lvflist[i]->recs = 0;
	}

	available_lvf = NUM_INIT_LVFS;

	i = 0;


/* EXECUTION STAGE */

	/* Send initial request to server. */
	retval = send_test_request(request);

	if (retval == -2)
		send_end();

	/* Exicute driver loop. */

	sleep(2);

	while (mode) {
	/* Get a reply and/or LVF from the in queue */
		/* Check open LVFs before a reply is fetched.... */
		
		lvfopen = ddc_get_LVFs_pending();

		if (nomesg && !lvfopen) 
		{
			repret = ddc_get_reply(&reply, &lvf,
				&lvf2, DDC_WAIT); 
		}
		else
		{
			repret = ddc_get_reply(&reply, &lvf,
				&lvf2, DDC_NOWAIT);
		}

		if (!repret)
			nomesg = 0;
		else
			nomesg = 1;

		/* Act on reply and/or LVF */
		switch (repret) {
		case 2:		/* handle reply and LVF */
		{
			fprintf( stderr, "Reply and LVF came in\n");

			if (transaction_action(reply->trans_key,
						TRAN_ACTION_LOOKUP))
			{
				sprintf(temp, "Invalid reply received from\
					server.\n");
				write_mesg(DRIVER_ERR, temp);
				break;
			}

			show_reply_data(reply);
			if (reply->err_no == 1)
			{
				transaction_action(reply->trans_key,
						   TRAN_ACTION_REMOVE);
			}

			lvf_action(lvf->LVFname, LVF_ACTION_ADDLVF);
			break;
		}
		case 1:		/* handle reply only */
		{
			fprintf( stderr, "Reply came in\n");

			if (transaction_action(reply->trans_key,
						TRAN_ACTION_LOOKUP))
			{
				sprintf(temp, "Invalid reply received from\
					server.\n");
				write_mesg(DRIVER_ERR, temp);
				break;
			}

			show_reply_data(reply);  
			if (reply->err_no == 1 ||
				reply->err_no == 0)
			{
				transaction_action(reply->trans_key,
						   TRAN_ACTION_REMOVE);
			}

			break;
		}
		case 0:		/* no message to handle */
		{
			fprintf( stderr, "No message incoming\n");
			break;
		}
		case -1:	/* hard system error */
		{
			fprintf( stderr, "System error\n"); 
			raise(SIGTERM);
			break;
		}
		case -2:	/* handle 1 or 2 LVFs */
		{
			fprintf( stderr, "1 or 2 LVFs came in\n");
			if ((lvfopen - (ddc_get_LVFs_pending()-1)) == 0)
			{
				show_lvf_data( lvf );
				lvf_action(lvf->LVFname, LVF_ACTION_ADDLVF);
			}	
			/* Two lvfs came in */
			else
			{
				show_lvf_data( lvf );
				show_lvf_data( lvf2 );
				lvf_action(lvf->LVFname, LVF_ACTION_ADDLVF);
				lvf_action(lvf2->LVFname, LVF_ACTION_ADDLVF);
			}
			break;
		}
		case -3:	/* Stop driver, replies pending */
		{
			write_mesg(DRIVER_ERR, "\nServer wants to end now.\n\n");

			continue;
			break;
		}
		case -4:	/* Stop driver now */
		{
			write_mesg(DRIVER_ERR, "\nMessage queues empty.\n\n");
			mode = 0;	
			continue;
			break;
		}
		}

		/* Now read some records from the LVF queue */
		lvfopen = ddc_get_LVFs_pending();
		/* Get 1 LVF record for each pending...
			as long as LVFs pending */
		for( i=0; (i < (lvfopen * 2 ) && ddc_get_LVFs_pending()); i++ )
		{
			int	lvfret, tot_recs;
			LVF_HEADER *lvfins = ((LVF_HEADER *) 0);

			memset(LVFname, 0, 25);
			memset(LVFrecord, 0, max_size);
			memset(action, 0, 1);
	
			lvfret = ddc_get_LVFrecord( LVFname, LVFrecord,
				max_size, cur_size, action );
			switch (lvfret) {
			case 1: /* got DDC_LIST_ELEM */
				copy_LVF_element (LVFrecord, action,
						LVFname, lvf_rec );
		    		show_lvf_data_element(lvf_rec);
				lvf_action(LVFname, LVF_ACTION_ADDELEM);
				break;
			case 0: /* got DDC_LIST_END */
				/* Check correct # of recs received here */
				tot_recs = lvf_action(LVFname,
							LVF_ACTION_NUMRECS);
				if (tot_recs != atoi(LVFrecord))
				{
				    strcopy(lvf_rep->ddc_id, "*TERM_001");
				    strcopy(lvf_rep->LVFname, LVFname); 
				    strcopy(lvf_rep->dtime, "19950825111112"); 
				    lvf_rep->dtype = 0;
				    ddc_send_LVF_repeat(lvf_rep);
				    strcopy(lvf_rep->ddc_id, "");
				    strcopy(lvf_rep->LVFname, ""); 
				    strcopy(lvf_rep->dtime, ""); 
				}
				else
				{
					strcopy(action, "E");
					copy_LVF_element (LVFrecord, action,
						LVFname, lvf_rec );
		    			show_lvf_data_element(lvf_rec);
				}
				lvf_action(LVFname, LVF_ACTION_REMLVF);
				break;
			case -1: /* got nothing or bad message */
				break;
			}
		}
		/* Send some requests out - driver developer will insert
			logic to get requests from devices and put data
			into DDC_REQUEST structure
		*/
		/* Test procedure to show requests can be answered by server */
		if (open_trans < 15 && mode && !wait_for_inquiry)
		{
			retval = send_test_request(request);

			if (retval == -2)
				send_end();
		}

	} /* end while() */


/* CLEAN-UP STAGE */

	/* Clean up message queues. */

	shutdown_drv(LVFrecord, LVFname, action, max_size, cur_size, repret,
		 request, reply, lvf, lvf2);

	/* Free request list's memory. */
	for(i = 0; i < available_trans; i++)
	{
		if (reqlist[i])
			free(reqlist[i]);
	}

	free(reqlist);

	/* Free LVF list's memory. */
	for(i = 0; i < available_lvf; i++)
	{
		if (lvflist[i])
		{
			if (lvflist[i]->lvfname)
				free(lvflist[i]->lvfname);

			free(lvflist[i]);
		}
	}

	free( lvflist );

	/* Free valid function server table's memory. */
	hash_action(&item, HASH_ACTION_REMOVE);

	/* Free allocated pointer variable memory. */
		free(LVFname);
		free(LVFrecord);	
#ifndef _WIN32
		free(action);
#endif
		free(cur_size);	

	/* Let server know that driver is ready to stop. */
	ddc_ready_to_stop();
/*
	ddc_free_REQUEST_memory(request);
	ddc_free_REPLY_memory(reply);
	ddc_free_LVF_HEADER_memory(lvf);
	ddc_free_LVF_HEADER_memory(lvf2);
	ddc_free_LVF_REPEAT_memory(lvf_rep);
	ddc_free_LVF_FORM_memory(lvf_rec);
*/
} /* end process() */


int	main( argc, argv )
int	argc;
char	*argv[];

{
	MENU_LIST	*mlist;
	GROUPLIST	*glist = 0;
	TERMINALS	*valid_terms = 0;
	int			i = 0,a = 0;
	char 		*df, debug_file[256], buf[256], *envstr, test[1];
	char		helpvar1[256], helpvar2[256], helpvar3[256];
	struct		tm	*now;

	if ( argc >= 5 ) {
		strcopy(sepchar, argv[4]);
	}
	else
	{
		strcopy(sepchar, ",");
		fprintf(stderr, "Warning: arg[4] defaulted to ,\n");
		fflush(stderr);
	}

	initialize_global_variables();

	/* Catch signal 15 - clean up when it occurs */

#if ! defined _WIN32
	signal(SIGTERM, clean_kill);
#endif

	dummy_ptr = getenv("DDC_INITIATE");

	if (dummy_ptr) {
		ddc_initiate = atoi(dummy_ptr);
		fprintf(stderr,"DDC_INITIATE: %d\n",ddc_initiate);
	} else
		fprintf(stderr, "Warning: DDC_INITIATE set to default\n");	
	
	fflush(stderr);

	/* Open log file for debugging if DDC_TEST_DRIVER is set */
	df = &(debug_file[0]); 
	df = getenv("DDC_TEST_DRIVER");
	
	if (df)
	{
		fprintf(stderr,"DDC_TEST_DRIVER: %s\n", df);
		fflush(stderr);

		logvar = LOGGING_ON; 
		debug_fd = fopen(df, "a+");

		/* Set time of test run - write to file		*/ 
		get_time(&now);

        	sprintf(buf, "%02d-%02d-%02d[%02d:%02d:%02d]%s",
		now->tm_year % 100, now->tm_mon+1, now->tm_mday,
		now->tm_hour, now->tm_min, now->tm_sec, "Start test\n");
		fwrite(buf, strlen(buf)+1, 1, debug_fd);
		fflush(debug_fd);
	} else {
		fprintf(stderr, "Warning: DDC_TEST_DRIVER not set\n");	
		fflush(stderr);
	}
	
	dummy_ptr = getenv("DDC_LOGGING");

	if (dummy_ptr) 
	{
		logvar = atoi(dummy_ptr);
		
		switch( logvar ) {
			case 0:
			case 1:
			case 2: {
				ddc_logging = logvar;
				fprintf(stderr,"DDC_LOGGING: %d\n",ddc_logging);
				break;
			}
			default: {
				ddc_logging = 0;
				fprintf(stderr, "Warning: DDC_LOGGING defaulting to 0\n");	

				break;
			}
			fflush(stderr);
		}
	}

	dummy_ptr = getenv( "DDC_MSGLEN" );
	
	if ( dummy_ptr == (char *) 0 )
	{
		allowed_msglen = 1023;
		fprintf(stderr, "Warning: DDC_MSGLEN defaulting to 1023\n");	
	} else {
		allowed_msglen = atoi( dummy_ptr ); 
		fprintf(stderr,"DDC_MSGLEN: %d\n", allowed_msglen);
	}
	fflush(stderr);

	/* Function to create dummy group list			*/
	ddc_groups_by_driver( &glist );

	/* Init stage start 					*/
	if (argc >= 2)
	{
		strcopy( helpvar1, argv[1] );
		fprintf(stderr,"argv[1]: %s\n",helpvar1);
	} else {		
		strcopy( helpvar1, "ddc.in" );
		fprintf(stderr, "Warning: arg[1] defaulted to %s\n",helpvar1);
	}
	fflush(stderr);

	if (argc >= 3)
	{
		strcopy( helpvar2, argv[2] );
		fprintf(stderr,"argv[2]: %s\n",helpvar2);
	} else {
		strcopy( helpvar2, "ddc.out" );
		fprintf(stderr, "Warning: arg[2] defaulted to %s\n",helpvar2);
	}
	fflush(stderr);

	if (argc >= 4)
	{
		strcopy( helpvar3, argv[3] );
		fprintf(stderr,"argv[3]: %s\n",helpvar3);
	} else {
		strcopy( helpvar3, "ddc.lvf");
		fprintf(stderr, "Warning: arg[3] defaulted to %s\n",helpvar3);
	}
	fflush(stderr);

	if ( ddc_set_separation( sepchar ) ) 
	{
		fprintf( stderr, "Driver In Queues incorrect \n" );
		fflush(stderr);
	}
    
	if ( ddc_create_driver_queues( helpvar2, helpvar3 ) ) 
	{
		fprintf( stderr, "Driver In Queues incorrect \n" );
		fflush(stderr);
	}
    
	switch( ddc_initiate ) 
	{
		case 0:
			break;
		case 1:
		{
			/* Example ddc_start_server executes program 'ddcserver6.1' */

			dummy_ptr = getenv("PWD");

			sprintf (dummy, "%s/ddcserver6.1", dummy_ptr);

			ddc_start_server(dummy, 
		                         helpvar1,
		                         helpvar2,
		                         helpvar3,
		                         sepchar,
		                         "drv_initiate",
				         "");
			break;
		}
		case 2: 
		{
			/* Example ddc_start_server executes program 'otuddc1206' */
			bse_dir = getenv("BSE");
			
			sprintf (dummy, "%s/bin/ba6.1", bse_dir);
		
			ddc_start_server(dummy,
		                         "otuddc1206",
		                         "ddcdriver6.1",
		                         helpvar1,
		                         helpvar2,
		                         helpvar3,
				         NULL);
			break;
		}
		case 3: 
		{
			/* Example ddc_start_bshell executes program 'otuddc1206'  */

	                dummy_ptr = getenv("DDC_START_TEST");
			
			sprintf (driver_name, "%s", argv[0]);
			fprintf(stderr,"driver_name: %s\n",driver_name);
			fflush(stderr);

			if (dummy_ptr)
			{
				/* tuddc1206 started manually for testing */

				fprintf( stderr, "otuddc1206 %s %s %s %s %s\n", 
						  "ddcdriver6.1",
						  helpvar1,
						  helpvar2,
						  helpvar3,
						  NULL);

			      	fprintf( stderr, "Start tuddc1206 and Hit Enter\n");

				gets(test);
			}
			else
			{
				if (ddc_start_bshell("otuddc1206", 
		                            "ddcdriver6.1", 
		                            helpvar1, 
		                            helpvar2, 
		                            helpvar3, 
		                            NULL)) 
				{
					fprintf(stderr,"ddc_start_bshell() failed.\n");
					exit(1);
				}
			}
			break;
		}
		default: 
		{
			break;
		}
	}

	if ( ddc_open_server_queue( helpvar1, ddc_initiate ) ) 
	{
		fprintf( stderr, "Driver Out Queue incorrect \n" );
	}

	if ( ddc_init_mem( ) ) 
	{
		fprintf( stderr, "Memory Initialisation is incorrect \n" );
	}

	if (ddc_initiate < 2)
	{

	  if ( ddc_group_selection( glist ) ) 
	  {
		fprintf( stderr, "Group selection is incorrect \n" );
	  }
	
	  if ( ddc_get_menus_functions( ) ) 
	  {
		fprintf( stderr, "Menus&functions is incorrect \n" );
	  }

	}

	/* Init stage end				*/

	/* Print Menu structure to stdout */
	print_menu_tree( glist ); 
	
	/* Get the list of menus for the groups here and free the memory
		This is done to save resources for the "dummy" driver	*/

	mlist = ddc_get_menu_list();
	i = ddc_free_menu_memory(mlist); 
	/* Create list of terminals for testing purposes - DDC driver should
		poll for devices and populate TERMINALS data structure
	*/

	/* Function to create dummy terminal list	*/
	create_term_list(&valid_terms);

	if ( ddc_terminal_list (valid_terms)) {
		FILELINE;
	}
	else {
		fprintf( stderr, "Terminals failed to initialise");
	}

	/* Set up a loop that will allow the debugger to be started while
	   running with the real server.                                  */
		
	envstr = getenv("DDC_USE_DEBUGGER");

	if (envstr)
	{
		a = atoi(envstr);
		while(a){}
	}
	else
		fprintf(stderr, "Warning:  DDC not using debugger\n");

	/* Find out how many input files are going to be used and open them. */
	
	envstr = getenv("DDC_NUM_INFILES");
	
	if (!envstr)
	{
		fprintf(stderr, "Error:  Environment variable DDC_NUM_INFILES not set\n");
		readreq = -1;
		exit(1);
	} 
	else 
	{

		if (!strcmp(envstr, "0"))
			readreq = -1;
		else {
			num_infiles = atoi(envstr);
			
			readreq = open_input_files(num_infiles, "d_infile");
			if (readreq == -1)
			{
				fprintf(stderr, "Error:  Too many input files. \n\n"); 
				exit(2);
			}
			if (readreq == 1)
			{
				fprintf(stderr, "Error: Can not open input files. \n\n");
				exit(1);
			}
			lastread = 0;
		}
	}

	/* Find out how many requests should be sent before shutting down. */

	envstr = getenv("DDC_NUM_REQS");
	
	if (!envstr)
	{
		fprintf(stderr, "Error:  Environment variable DDC_NUM_REQS ");
		fprintf(stderr, "not set.\n\n");
		num_requests = 0;
		exit(1);
	} 
	else 
	{
		num_requests = atoi(envstr);
		fprintf(stderr, "num_requests = %d\n", num_requests);
	}

	/* Find out if we are running in online or offline mode. */
	envstr = getenv("DDC_TRANS_MODE");

	if (!envstr)
	{
		fprintf(stderr, "Warning:  Environment variable DDC_TRANS_MODE not set ");
		transaction_mode = 1;
	} 
	else
	{
		transaction_mode = atoi(envstr);
		fprintf(stderr, "transaction_mode = %d\n\n", transaction_mode);
	}

	if ( ! ddc_start_communications( ) ) {
		process();
		FILELINE;
	}
	else
	{
	/* If we get here the server is down or wants to stop or never got
		initialized properly					*/
	
		fprintf( stderr, "Initialisation is incorrect \n" );
		/* Initialisation is correct */
	
		ddc_stop_server();

		ddc_ready_to_stop();			
	}

	/* DDC driver should log off devices, clean up, etc. here. To be 
		defined by driver implementation			*/

	if (df && strlen(df))
	{
		get_time(&now);

        	sprintf(buf, "%02d-%02d-%02d[%02d:%02d:%02d]%s",
			now->tm_year % 100, now->tm_mon+1, now->tm_mday,
			now->tm_hour, now->tm_min, now->tm_sec, "End test\n");
		fwrite(buf, strlen(buf)+1, 1, debug_fd);
		fflush(debug_fd);
		fclose(debug_fd);
	}

	close_input_files(num_infiles);

	return( 0 );

} /* end main() */
