-- **************************************************************************
-- *
-- * File	: cnvdate.sql	( for BaanIV )
-- *
-- * Purpose	: Convert representation of the empty Baan date.
-- *
-- *		  The Baan/Oracle driver has a resource that controls the empty
-- *		  date representation in the Oracle database.
-- *			ora_date:0 (default) Jan. 1st -9999
-- *			ora_date:1           Jan. 1st -4712
-- *		  
-- *		  The Baan-Oracle driver uses the date Jan. 1st -9999 as the
-- *		  representation of the empty Baan date.
-- *		  When setting the ora_date:1 resource in the $BSE/lib/defaults/db_resource
-- *		  file, the empty date representation is Jan. 1st -4712.
-- *
-- *		  This is done because PL/SQL, SQL*Plus and SQL*Loader are not able
-- *		  to handle dates before Jan. 1st -4712.
-- *		  
-- *		  In BaanERP 5.0 and higher only the Jan. 1st -4712 representation is used.
-- *		  
-- * Author	: Anthon Ouwendijk, Baan Tech	( aouwendijk@baan.nl )
-- * Date	: May 1998
-- *
-- **************************************************************************
-- *
-- *	-->	This script must be run from SQL*Plus. Connect as 'SYS'
-- *
-- *	-->	This script assumes company 0 (Tools) is in Oracle. If not modify
-- *		the query that selects the tables to be converted.
-- *
-- *	!!!!!!!!!!!! WARNING !!!!!!!!!!!!
-- *
-- *	-->	The database tables of ALL Baan owners are updated by this script
-- *
-- *	-->	Do not forget to set ora_date:1 in $BSE/lib/defaults/db_resource
-- *
-- **************************************************************************

set verify	off;
set feedback	off;
set line	200
set heading	off

-- **************************************************************************
--
-- Variables used by the procedure LogMesg() to generate unique log id's
-- Table BaanConvertDateLog contains the log messages generated by this script.
-- The messages of the latest run are shown to the user after completion.
--

CREATE	TABLE	BaanConvertDateLog( id	VARCHAR2( 25 ), message VARCHAR2( 125 ) )
/

VARIABLE	LogTim	VARCHAR2( 15 )
VARIABLE	LogNum	NUMBER
BEGIN
		SELECT	TO_CHAR( SYSDATE, 'YYMMDD HH24:MI:SS' ) INTO :LogTim from DUAL;
		:LogNum := 1;
END;
/

-- **************************************************************************
--
-- PL/SQL script starts here
--
DECLARE
	fatal_problem	EXCEPTION;	-- When raised stop conversion

	-- select all BaanERP tables / columns that need conversion

	CURSOR	c_tab
	IS
		SELECT	owner || '.' || table_name	tab,
			column_name			col
		FROM	all_tab_columns
		WHERE	data_type = 'DATE'
		AND	table_name like 'T___________'
		AND	owner IN (				-- only BaanIV/ERP owners
			SELECT	distinct owner FROM all_tab_columns
			WHERE	table_name = 'TTTADV990000'
		)
	;


	PROCEDURE LogMesg( msg CHAR )
	IS
		LogId	VARCHAR2( 25 );
	BEGIN
		LogId := :LogTim || TO_CHAR( :LogNum , '09999' );

		:LogNum := :LogNum + 1;

		INSERT INTO BaanConvertDateLog
		VALUES	( LogId, msg );
		commit;
	END;


	--
	-- The conversion procedure that does the job
	--

	PROCEDURE BaanConvertDate ( tab CHAR, col CHAR )
	IS
		cid	INTEGER;
		nrows	INTEGER;
		stmt	VARCHAR2( 1024 );
	BEGIN
		--
		-- Update date column. Set value to Jan 1st 4712 BC (empty Baan date)
		-- when the date is less (this should be the old empty date Jan 1st 9999 BC).
		stmt := 'UPDATE ' || tab
			|| ' SET '   || col || ' = TO_DATE( ''1 0'', ''J SSSSS'' )' 
			|| ' WHERE ' || col || ' < TO_DATE( ''1 0'', ''J SSSSS'' )'
		;

		cid := dbms_sql.open_cursor;

		dbms_sql.parse( cid, stmt, dbms_sql.v7 );
		nrows := dbms_sql.execute( cid );

		IF	nrows > 0
		THEN
			commit;
			LogMesg( tab || ' - ' || col || ' converted:     ' || TO_CHAR( nrows, 'B9999' ) || ' records changed.' );
		ELSE
			rollback;
			LogMesg( tab || ' - ' || col || ' checked.' );
		END IF;

		dbms_sql.close_cursor( cid );

	EXCEPTION
		WHEN OTHERS THEN
			dbms_sql.close_cursor( cid );

			LogMesg( tab || ' - ' || col || ' not converted: ' || SQLERRM( SQLCODE ) );
			rollback;
	END;

	PROCEDURE check_package 
	IS
		package_status	VARCHAR2( 20 );
		wrong_state	EXCEPTION;
	BEGIN
		SELECT	status INTO package_status
			FROM	all_objects
			WHERE	OBJECT_NAME = 'DBMS_SQL'
			AND	OBJECT_TYPE = 'PACKAGE'
		;

		IF package_status != 'VALID'
		THEN 
			RAISE wrong_state;
		END IF;
	EXCEPTION
		WHEN NO_DATA_FOUND THEN
			LogMesg( 'ERROR: package dbms_sql not installed.' );
			LogMesg( '       Please install $ORACLE_HOME/rdbms/admin/dbmssql.sql' );
			RAISE fatal_problem;

		WHEN wrong_state THEN
			LogMesg( 'ERROR: package dbms_sql has state: ' || package_status );
			LogMesg( '       Please run ''alter package dbms_sql compile'' as ''SYS''' );
			RAISE fatal_problem;
	END;

BEGIN
	--
	-- The entry point.
	-- Initialization and the conversion loop
	--

	LogMesg( 'BaanERP Oracle date conversion script started by ' || USER );

	IF	USER != 'SYS'
	THEN
		LogMesg( 'ERROR: only Oracle user ''SYS'' is allowed to run this script.' );
		LogMesg( '       Please connect as ''SYS''' );
		RAISE fatal_problem;
	END IF;
	
	check_package;

	FOR tab_rec IN c_tab
	LOOP
		BaanConvertDate( tab_rec.tab, tab_rec.col );
	END LOOP;

	LogMesg( 'The conversion log has been saved in table ''sys.BaanConvertDateLog''' );
	LogMesg( TO_CHAR( SYSDATE, 'YYMMDD HH24:MI:SS' ) || '   conversion finished.' );

EXCEPTION
	WHEN  fatal_problem THEN
		LogMesg( TO_CHAR( SYSDATE, 'YYMMDD HH24:MI:SS' ) || ': conversion stopped due to fatal error.' );

	WHEN OTHERS THEN
		RAISE;
END;
/

SELECT		*
FROM		BaanConvertDateLog
WHERE		id >= :LogTim
ORDER BY	id
/
