/**
 * Begin Copyright Notice
 *
 * NOTICE
 *
 * THIS SOFTWARE IS THE PROPERTY OF AND CONTAINS CONFIDENTIAL INFORMATION OF
 * INFOR AND/OR ITS AFFILIATES OR SUBSIDIARIES AND SHALL NOT BE DISCLOSED
 * WITHOUT PRIOR WRITTEN PERMISSION. LICENSED CUSTOMERS MAY COPY AND ADAPT
 * THIS SOFTWARE FOR THEIR OWN USE IN ACCORDANCE WITH THE TERMS OF THEIR
 * SOFTWARE LICENSE AGREEMENT. ALL OTHER RIGHTS RESERVED.
 *
 * (c) COPYRIGHT 2017 INFOR. ALL RIGHTS RESERVED. THE WORD AND DESIGN MARKS
 * SET FORTH HEREIN ARE TRADEMARKS AND/OR REGISTERED TRADEMARKS OF INFOR
 * AND/OR ITS AFFILIATES AND SUBSIDIARIES. ALL RIGHTS RESERVED. ALL OTHER
 * TRADEMARKS LISTED HEREIN ARE THE PROPERTY OF THEIR RESPECTIVE OWNERS.
 *
 * End Copyright Notice
 */
package com.infor.ln.wb.common.shared.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import com.infor.ln.wb.common.shared.model.LNTable.Field;


/**
 * A link between two {@link LNTable}s.
 */
public class LNReference {
    private LNTable mFromTable;
    private Field mFromField;
    private LNTable mToTable;
    private boolean mIsFromReference;

    /**
     * Instantiate a new {@link LNReference}.
     * @param fromTable the {@link LNTable} that contains the reference.
     * @param fromField the {@link Field} that refers to the other table.
     * @param toTable the {@link LNTable} to which the other table refers.
     * @param isFromReference indicates whether this reference is functionally turned around.
     */
    public LNReference(LNTable fromTable, Field fromField, LNTable toTable, boolean isFromReference) {
        assert fromField.getRefersToTableName() != null && fromField.getRefersToTableName().equals(toTable.getCode());
        mFromTable = fromTable;
        mFromField = fromField;
        mToTable = toTable;
        mIsFromReference = isFromReference;
    }

    /**
     * @return whether the reference is a from-reference.
     */
    public boolean isFromReference() {
        return mIsFromReference;
    }

    /**
     * @return the {@link LNTable} at which the reference starts.
     */
    public LNTable getFromTable() {
        return mFromTable;
    }

    /**
     * @return the {@link Field} that has the reference.
     */
    public Field getFromField() {
        return mFromField;
    }

    /**
     * @return the {@link LNTable} to which the reference points.
     */
    public LNTable getToTable() {
        return mToTable;
    }

    /**
     * @return a new {@link LNReference} that points in the other direction.
     */
    public LNReference invert() {
        return new LNReference(mFromTable, mFromField, mToTable, !mIsFromReference);
    }

    @Override
    public String toString() {
        return mIsFromReference
                ? mToTable.getCode() + "<" + mFromTable.getCode() + "." + mFromField.getCode()
                : mFromTable.getCode() + "." + mFromField.getCode() + ">" + mToTable.getCode();
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((mFromField == null) ? 0 : mFromField.hashCode());
        result = prime * result + ((mFromTable == null) ? 0 : mFromTable.hashCode());
        result = prime * result + ((mToTable == null) ? 0 : mToTable.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof LNReference)) {
            return false;
        }
        LNReference other = (LNReference) obj;
        return mIsFromReference == other.mIsFromReference &&
                mFromField.equals(other.mFromField) &&
                mFromTable.equals(other.mFromTable) &&
                mToTable.equals(other.mToTable);
    }


    /**
     * A combination of multiple {@link LNReference}s to link between two {@link LNTable}s with
     * (optionally) multiple tables in-between.
     */
    public static class ReferencePath implements Iterable<LNReference> {
        private final List<LNReference> mRefs = new ArrayList<>();

        /**
         * Instantiate a new empty {@link ReferencePath}.
         */
        public ReferencePath() {
        }

        /**
         * Instantiate a new {@link ReferencePath} containing all references contained in the given path.
         * @param refs the {@link ReferencePath} to copy.
         */
        public ReferencePath(ReferencePath refs) {
            this();
            add(refs);
        }

        /**
         * Instantiate a new {@link ReferencePath} containing all {@link LNReference} in the given list.
         * @param references a {@link List} of {@link LNReference} to be added to the path.
         */
        public ReferencePath(List<LNReference> references) {
            for (LNReference ref : references) {
                inject(ref);
            }
        }

        /**
         * @return true iff the {@link ReferencePath} contains no {@link LNReference}s.
         */
        public boolean isEmpty() {
            return mRefs.isEmpty();
        }

        /**
         * @return the number of {@link LNReference}s in this path.
         */
        public int size() {
            return mRefs.size();
        }

        /**
         * Add all references contained in the given path to this path. This path must be empty.
         * @param refs the {@link ReferencePath} to copy.
         */
        public void add(ReferencePath refs) {
            assert isEmpty();
            // No need for validation, as refs must be valid already
            mRefs.addAll(refs.mRefs);
        }

        /**
         * Push an {@link LNReference} to the front of the path.
         * @param ref the {@link LNReference} to add.
         */
        public void push(LNReference ref) {
            assert !hasMixedDirections(ref, true);
            mRefs.add(0, ref);
        }

        /**
         * Inspect the first {@link LNReference} in the path.
         * @return the {@link LNReference} at the start of the path.
         */
        public LNReference first() {
            return mRefs.get(0);
        }

        /**
         * Pop the first {@link LNReference} from the path.
         * @return the {@link LNReference} at the start of the path.
         */
        public LNReference pop() {
            return mRefs.remove(0);
        }

        /**
         * Append an {@link LNReference} to the end of the path.
         * @param ref the {@link LNReference} to append.
         */
        public void inject(LNReference ref) {
            assert !hasMixedDirections(ref) : "ReferencePath has mixed references";
            assert isEmpty() || referencesConnect(last(), ref) : "New reference does not connect to last reference";
            mRefs.add(ref);
        }

        /**
         * Inspect the last {@link LNReference} in the path.
         * @return the {@link LNReference} at the end of the path.
         */
        public LNReference last() {
            return mRefs.get(mRefs.size() - 1);
        }

        /**
         * Eject the last {@link LNReference} from the path.
         * @return the {@link LNReference} at the end of the path.
         */
        public LNReference eject() {
            return mRefs.remove(mRefs.size() - 1);
        }

        /**
         * Make this {@link ReferencePath} a path containing only the last to-part (i.e. the path within the dataset).
         */
        public void makeToReference() {
            Iterator<LNReference> iter = mRefs.iterator();
            while (iter.hasNext()) {
                if (iter.next().isFromReference()) {
                    iter.remove();
                }
            }
        }

        /**
         * Create a reference path string, as used in the dataset. The current {@link ReferencePath} must consist of
         * only to-references.
         * @return a comma-separated string of from-tables and fields.
         */
        public String toRefString() {
            assert isEmpty() || !first().isFromReference() : "ReferencePath contains from-references";

            StringBuilder sb = new StringBuilder();
            for (LNReference ref : mRefs) {
                if (sb.length() > 0) {
                    sb.append(",");
                }
                sb.append(ref.getFromTable().getCode());
                sb.append(".");
                sb.append(ref.getFromField().getCode());
            }
            return sb.toString();
        }

        private boolean hasMixedDirections(LNReference newRef, boolean push) {
            if (isEmpty()) {
                return false;
            }
            if (!push) {
                // Appending a from-reference to a to-reference is invalid
                return newRef.isFromReference() ? !last().isFromReference() : false;
            } else {
                // Prepending a to-reference to a from-reference is invalid
                return newRef.isFromReference() ? false : first().isFromReference();
            }
        }

        private boolean hasMixedDirections(LNReference newRef) {
            return hasMixedDirections(newRef, false);
        }

        private static boolean referencesConnect(LNReference a, LNReference b) {
            String firstTable = a.isFromReference() ? a.getFromTable().getCode() : a.getToTable().getCode();
            String secondTable = b.isFromReference() ? b.getToTable().getCode() : b.getFromTable().getCode();
            return firstTable.equals(secondTable);
        }

        @Override
        public Iterator<LNReference> iterator() {
            return Collections.unmodifiableList(mRefs).iterator();
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            for (LNReference ref : mRefs) {
                if (sb.length() > 0) {
                    sb.append(",");
                }
                sb.append(ref.toString());
            }
            return sb.toString();
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + mRefs.hashCode();
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof ReferencePath)) {
                return false;
            }
            ReferencePath other = (ReferencePath) obj;
            return mRefs.equals(other.mRefs);
        }
    }
}