/**
 * Begin Copyright Notice
 *
 * NOTICE
 *
 * THIS SOFTWARE IS THE PROPERTY OF AND CONTAINS CONFIDENTIAL INFORMATION OF
 * INFOR AND/OR ITS AFFILIATES OR SUBSIDIARIES AND SHALL NOT BE DISCLOSED
 * WITHOUT PRIOR WRITTEN PERMISSION. LICENSED CUSTOMERS MAY COPY AND ADAPT
 * THIS SOFTWARE FOR THEIR OWN USE IN ACCORDANCE WITH THE TERMS OF THEIR
 * SOFTWARE LICENSE AGREEMENT. ALL OTHER RIGHTS RESERVED.
 *
 * (c) COPYRIGHT 2017 INFOR. ALL RIGHTS RESERVED. THE WORD AND DESIGN MARKS
 * SET FORTH HEREIN ARE TRADEMARKS AND/OR REGISTERED TRADEMARKS OF INFOR
 * AND/OR ITS AFFILIATES AND SUBSIDIARIES. ALL RIGHTS RESERVED. ALL OTHER
 * TRADEMARKS LISTED HEREIN ARE THE PROPERTY OF THEIR RESPECTIVE OWNERS.
 *
 * End Copyright Notice
 */
package com.infor.ln.wb.common.client.view;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.Style.Unit;
import com.google.gwt.event.dom.client.BlurEvent;
import com.google.gwt.event.dom.client.BlurHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.i18n.client.LocaleInfo;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.PopupPanel.PositionCallback;
import com.google.gwt.user.client.ui.ResizeComposite;
import com.google.gwt.user.client.ui.ScrollPanel;
import com.google.gwt.user.client.ui.Widget;
import com.google.web.bindery.event.shared.EventBus;
import com.google.web.bindery.event.shared.HandlerRegistration;
import com.google.web.bindery.event.shared.SimpleEventBus;
import com.infor.component.client.widgets.Button;
import com.infor.component.client.widgets.DropdownListbox;
import com.infor.component.client.widgets.ListItem;
import com.infor.component.client.widgets.LoadingIndicator;
import com.infor.component.client.widgets.SvgIcon;
import com.infor.component.client.widgets.TextInputField;
import com.infor.component.client.widgets.TriggerInputField;
import com.infor.component.client.widgets.container.HeaderLayoutPanel;
import com.infor.component.client.widgets.events.StateChangeEvent;
import com.infor.component.client.widgets.trees.SimpleTree;
import com.infor.component.client.widgets.trees.SimpleTree.Item;
import com.infor.ln.wb.common.client.WbCommon;
import com.infor.ln.wb.common.client.events.TableSelectionChangeEvent;
import com.infor.ln.wb.common.client.presenter.ITableTreePresenter;
import com.infor.ln.wb.common.client.presenter.ITableTreeView;
import com.infor.ln.wb.common.client.utils.SortMode;
import com.infor.ln.wb.common.client.utils.WBRtlUtils;
import com.infor.ln.wb.common.shared.model.LNReference;
import com.infor.ln.wb.common.shared.model.LNReference.ReferencePath;
import com.infor.ln.wb.common.shared.model.LNTable;
import com.infor.ln.wb.common.shared.resources.CommonLNLabels;
import com.infor.ln.wb.common.shared.resources.CommonLNMessages;


/**
 * TableTreeView is a view for table references.
 */
public class TableTreeView extends ResizeComposite implements ITableTreeView {
    private static final CommonLNLabels LABELS = WbCommon.labels();
    private static final CommonLNMessages MESSAGES = WbCommon.messages();
    private final EventBus mEventBus = new SimpleEventBus();
    private ScrollPanel mScrollPanel;
    private SimpleTree mTree = new SimpleTree();
    private ITableTreePresenter mPresenter;
    private SortMode mSortMode;
    private SortedReferenceBag mReferenceBag = new SortedReferenceBag();
    private DropdownListbox mSortSelection;
    private TriggerInputField mFilterField;
    private Loading mLoading = new Loading();
    private Button mStopFilterButton;
    private TextInputField mFilterInputField;


    private ClickHandler mTableClickHandler = new ClickHandler() {
        @Override
        public void onClick(ClickEvent event) {
            assert event.getSource() instanceof Item;
            Item item = (Item) event.getSource();

            assert item.getUserData() instanceof TTVUserData;
            TTVUserData ud = (TTVUserData) item.getUserData();

            LNTable selectedTable = ud.table;

            ReferencePath path = determineReferencePathForItem(item);

            mEventBus.fireEvent(new TableSelectionChangeEvent(selectedTable, path));
        }
    };

    private StateChangeEvent.Handler tablestatechangehandler = new StateChangeEvent.Handler() {
        @Override
        public void onStateChange(StateChangeEvent event) {
            assert event.getSource() instanceof Item;
            final Item src = (Item) event.getSource();

            assert src.getUserData() instanceof TTVUserData;
            final TTVUserData ud = (TTVUserData) src.getUserData();
            if (ud.registration == null) {
                // The handler is not removed while this event is being handled; adding the child items triggers
                // the same event again.
                return;
            }

            ud.registration.removeHandler();
            ud.registration = null;

            if (ud.isFromReference) {
                mPresenter.addFromReferences(ud.table, determineReferencePathForItem(src));
            } else {
                mPresenter.addToReferences(ud.table, determineReferencePathForItem(src));
            }
        }
    };

    /**
     * Instantiate a new (empty) {@link TableTreeView}.
     */
    public TableTreeView() {
        this(true);
    }

    /**
     * Instantiate a new (empty) {@link TableTreeView}.
     * @param useFilter whether or not to create the filter controls.
     */
    public TableTreeView(boolean useFilter) {
        HeaderLayoutPanel panel = new HeaderLayoutPanel();
        panel.setSize("100%", "100%");
        panel.setStylePrimaryName("TableTreeView");

        initWidget(panel);

        Widget headerWidget = createHeaderWidget();
        if (useFilter) {
            panel.setHeaderWidget(headerWidget);
        }
        panel.setContentWidget(createContentWidget());
    }

    @Override
    public void setPresenter(ITableTreePresenter presenter) {
        mPresenter = presenter;
    }

    /**
     * Set the current sort option.
     * @param order the {@link SortMode} that should be applied.
     */
    public void setTableSortOrder(SortMode order) {
        mSortMode = order;
        mSortSelection.selectItem(order.ordinal());
    }

    /**
     * @return the current sort option.
     */
    public SortMode getTableSortOrder() {
        return mSortMode;
    }

    private Widget createHeaderWidget() {
        FlowPanel header = new FlowPanel();
        header.setStylePrimaryName("TableTreeViewHeader");

        header.add(createSortWidget());

        header.add(createFilterWidget());
        header.add(createStopFilterButton());

        return header;
    }

    private Widget createSortWidget() {
        mSortSelection = new DropdownListbox();
        mSortSelection.getElement().getStyle().setWidth(150, Unit.PX);
        ListItem[] items = new ListItem[SortMode.values().length];
        for (SortMode mode : SortMode.values()) {
            items[mode.ordinal()] = new ListItem(mode.getDisplayName());
        }
        mSortSelection.setItems(items);
        mSortSelection.addValueChangeHandler(new ValueChangeHandler<Integer>() {
            @Override
            public void onValueChange(ValueChangeEvent<Integer> event) {
                mSortMode = SortMode.values()[event.getValue()];
                sortTree();
            }
        });
        mSortMode = SortMode.values()[1];
        mSortSelection.selectItem(1);

        return mSortSelection;
    }

    private Widget createFilterWidget() {
        mFilterInputField = new TextInputField();

        mFilterField = new TriggerInputField(mFilterInputField, new SvgIcon("icon-search"));
        mFilterInputField.addValueChangeHandler(new ValueChangeHandler<String>() {
            @Override
            public void onValueChange(ValueChangeEvent<String> event) {
                mPresenter.applyFilter(event.getValue().toLowerCase());
            }
        });
        mFilterInputField.addBlurHandler(new BlurHandler() {
            @Override
            public void onBlur(BlurEvent event) {
                mPresenter.applyFilter(mFilterInputField.getText().toLowerCase());
            }
        });
        mFilterField.addStyleName("FilterInput");
        return mFilterField;
    }

    private Widget createStopFilterButton() {
        mStopFilterButton = new Button();
        mStopFilterButton.addStyleName("Cancel");
        mStopFilterButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                mPresenter.stopFilter();
            }
        });
        return mStopFilterButton;
    }

    private Widget createContentWidget() {
        mScrollPanel = new ScrollPanel();
        mScrollPanel.setSize("100%", "100%");

        mTree.setMultiSelect(false);
        mTree.setSingleClickExpand(false);
        mScrollPanel.add(mTree);

        return mScrollPanel;
    }

    /**
     * @return content widget
     */
    public Widget getContentWidget() {
        return mScrollPanel;
    }

    @Override
    public void clear() {
        mTree.clear();
    }

    @Override
    public void addMainTable(LNTable table) {
        Item root = createMainTableItem(table, true);
        mTree.getRoot().addItem(root);
        if (root.getItemCount() == 0) {
            root.setFolder(false);
        } else {
            root.setExpanded(true);
        }
    }

    @Override
    public void addTables(ReferencePath commonPath, boolean fromRefs, List<LNReference> tables) {
        if (mTree.getRoot().getItemCount() == 0) {
            assert false : "Main table item must be inserted before adding tables";
            return;
        }

        Item commonParent = findCommonParent(commonPath, fromRefs);
        if (commonParent == null) {
            return;
        }

        assert commonParent.getItemCount() == 1 && commonParent.getItem(0).getUserData() == null
                : "Only the loading indicator should be present for this item";
        commonParent.removeItem(0);

        if (tables.isEmpty()) {
            commonParent.setFolder(false);
        } else {
            mReferenceBag.reset(mSortMode, fromRefs);
            for (LNReference entry : tables) {
                mReferenceBag.add(entry);
            }

            for (LNReference ref : mReferenceBag) {
                commonParent.addItem(createTableItem(ref, true));
            }
            TTVUserData ud = (TTVUserData) commonParent.getUserData();
            if (ud.type != ItemType.MAINSUB) {
                if (ud.registration == null) {
                    commonParent.setExpanded(true);
                } else {
                    // adding tables to a parent whose handler is still present - i.e. the presenter triggered the
                    // non-lazy loading
                    ud.registration.removeHandler();
                    ud.registration = null;
                }
            }
        }
    }

    @Override
    public void addMainTableForFilter(LNTable table) {
        Item root = createMainTableItem(table, false);
        mTree.getRoot().addItem(root);
    }

    @Override
    public void addTableForFilter(ReferencePath tablePath, LNReference tableRef) {
        if (tableRef == null) {
            mTree.getRoot().getItem(0).addStyleName("filtermatch");
            return;
        }

        Item parentItem;
        if (tablePath.isEmpty()) {
            parentItem = findOrCreateReferenceSubfolder((Item) mTree.getRoot().getItem(0), tableRef.isFromReference());
        } else {
            parentItem = findItemForFilter(tablePath, tableRef);
        }

        Item tableItem = findItemForReference(parentItem, tableRef, true);
        if (tableItem == null) {
            tableItem = createTableItem(tableRef, false);
            parentItem.addItem(tableItem);
            parentItem.setExpanded(true);
        }
        tableItem.addStyleName("filtermatch");
    }

    private Item findCommonParent(ReferencePath commonPath, boolean fromRefs) {
        assert mTree.getRoot().getItemCount() > 0 : "Main table must be added before adding tables";

        Item commonParent = null;
        if (commonPath.isEmpty()) {
            commonParent = (Item) mTree.getRoot().getItem(0);
        } else {
            commonParent = findItemForReferencePath(commonPath);
            assert commonParent != null : "Failed to find common parent while adding tables";
        }
        if (commonParent == null) {
            return null;
        }

        commonParent = determineReferenceSubfolder(commonParent, fromRefs);

        return commonParent;
    }

    private Item determineReferenceSubfolder(Item parent, boolean fromRefs) {
        TTVUserData parentUD = (TTVUserData) parent.getUserData();
        switch (parentUD.type) {
        case MAINTABLE:
        case TABLEFOLDER:
            return (Item) parent.getItem(parent.getItemCount() == 2 && fromRefs ? 1 : 0);
        default:
            return parent;
        }
    }

    private Item findOrCreateReferenceSubfolder(Item parent, boolean fromRefs) {
        TTVUserData parentUD = (TTVUserData) parent.getUserData();
        switch (parentUD.type) {
        case MAINTABLE:
        case TABLEFOLDER:
            for (int i = 0; i < parent.getItemCount(); ++i) {
                if (((TTVUserData) parent.getItem(i).getUserData()).isFromReference == fromRefs) {
                    return (Item) parent.getItem(i);
                }
            }
            Item folderItem = createReferenceFolder(parentUD.table, fromRefs, false);
            parent.addItem(folderItem,
                    parent.getItemCount() == 1 && fromRefs ? 1 : 0,
                    false);
            parent.setExpanded(true);
            return folderItem;
        default:
            return parent;
        }
    }

    @Override
    public HandlerRegistration addSelectionChangeHandler(TableSelectionChangeEvent.Handler handler) {
        return mEventBus.addHandler(TableSelectionChangeEvent.TYPE, handler);
    }

    @Override
    public void selectTable(LNTable table, ReferencePath tablePath) {
        Item item = findItemForReferencePath(tablePath);
        if (item != null) {
            mTree.setItemSelected(item, true, false, false);
            mScrollPanel.ensureVisible(item);
        }
    }

    private void sortTree() {
        if (mTree.getRoot().getItemCount() > 0) {
            sortTree((Item) mTree.getRoot().getItem(0), true);
        }
    }

    private void sortTree(Item parent, boolean force) {
        if (!parent.isFolder() || ( !parent.isExpanded() && !force )) {
            resetTreeItem(parent);
            return;
        }

        TTVUserData parentUD = (TTVUserData) parent.getUserData();
        if (parentUD.type == ItemType.TABLEFOLDER || parentUD.type == ItemType.MAINTABLE) {
            sortTree((Item) parent.getItem(0), false);
            if (parent.getItemCount() > 1) {
                sortTree((Item) parent.getItem(1), false);
            }
        } else {
            for (int iChild = 0; iChild < parent.getItemCount(); ++iChild) {
                if (parent.getItem(iChild).isFolder()) {
                    sortTree((Item) parent.getItem(iChild), false);
                }
            }
            if (parent.getItemCount() > 1) {
                sortChildren(parent);
            }
        }
    }

    private void sortChildren(Item parent) {
        TTVUserData parentUD = (TTVUserData) parent.getUserData();
        Map<LNReference, Item> itemmap = new HashMap<>();
        mReferenceBag.reset(mSortMode, parentUD.isFromReference);

        for (int iChild = 0; iChild < parent.getItemCount(); ++iChild) {
            Item child = (Item) parent.getItem(iChild);
            TTVUserData childUD = (TTVUserData) child.getUserData();
            mReferenceBag.add(childUD.reference);
            itemmap.put(childUD.reference, child);
        }

        parent.clearChildren();
        int iNew = 0;
        for (LNReference ref : mReferenceBag) {
            parent.addItem(itemmap.get(ref), iNew++, false);
        }
        parent.updateLines();
    }

    private void resetTreeItem(Item item) {
        TTVUserData ud = (TTVUserData) item.getUserData();
        switch (ud.type) {
        case TABLE:
        case MAINSUB:
        case SUBFOLDER:
            item.clearChildren();
            item.addItem(createSpinnyThingy());
            ud.registration = item.addStateChangeHandler(tablestatechangehandler);
            break;
        default:
            break;
        }
    }

    private ReferencePath determineReferencePathForItem(Item inItem) {
        TTVUserData ud;
        ReferencePath path = new ReferencePath();
        Item item = inItem;
        while (item != mTree.getRoot()) {
            assert item.getUserData() instanceof TTVUserData;
            ud = (TTVUserData) item.getUserData();

            if (!ud.isSubFolder && ud.reference != null) {
                path.push(ud.reference);
            }

            item = (Item) item.getParentItem();
        }
        return path;
    }

    private Item findItemForReferencePath(ReferencePath tablePath) {
        return findItemForReferencePath(tablePath, (Item) mTree.getRoot().getItem(0));
    }

    private Item findItemForFilter(ReferencePath tablePath, LNReference tableRef) {
        boolean foundParent = true;
        boolean parentHasFolders = true;
        Item parentItem = (Item) mTree.getRoot().getItem(0);

        for (LNReference ref : tablePath) {
            if (foundParent) {
                ReferencePath subPath = new ReferencePath();
                subPath.inject(ref);

                Item childItem;
                if (parentHasFolders) {
                    parentItem = findOrCreateReferenceSubfolder(parentItem, ref.isFromReference());
                }
                childItem = findItemForReference(parentItem, ref, true);

                if (childItem != null) {
                    parentItem = childItem;
                    if (!ref.isFromReference()) {
                        parentHasFolders = false;
                    }
                } else {
                    foundParent = false;
                }
            }
            if (!foundParent) {
                if (parentHasFolders) {
                    parentItem = findOrCreateReferenceSubfolder(parentItem, ref.isFromReference());
                }
                Item tmp = createTableItem(ref, false);
                parentItem.addItem(tmp);
                parentItem.setExpanded(true);
                parentItem = tmp;
                if (!ref.isFromReference()) {
                    parentHasFolders = false;
                }
            }
        }
        if (parentHasFolders) {
            parentItem = findOrCreateReferenceSubfolder(parentItem, tableRef.isFromReference());
        }
        return parentItem;
    }

    private Item findItemForReferencePath(ReferencePath tablePath, Item parent) {
        Item item = parent;
        for (LNReference ref : tablePath) {
            item = determineReferenceSubfolder(item, ref.isFromReference());
            boolean emptyItem = false;
            if (item.getItemCount() == 1 && item.getItem(0).getUserData() == null) {
                // The subfolder has not been filled yet. This happens when selecting a table that is part of the
                // dataset, after the tree has been reset (e.g. when changing the main table, and undo-ing that action).
                // Evil trick: expanding the item causes a state change, which causes the item to be filled; given
                // that the table is known to reside in the cache, all callbacks will be called before expansion
                // returns.
                emptyItem = true;
            }
            item.getParentItem().setExpanded(true);
            item.setExpanded(true);
            item = findItemForReference(item, ref, emptyItem); // Just in case the trick didn't work.
            if (item == null) {
                break;
            }
        }
        return item;
    }

    private Item findItemForReference(Item parent, LNReference ref, boolean forFilter) {
        Item item = parent;
        boolean found = false;
        for (int iItem = 0; iItem < item.getItemCount(); ++iItem) {
            Item child = (Item) item.getItem(iItem);
            assert child.getUserData() instanceof TTVUserData;
            TTVUserData ud = (TTVUserData) child.getUserData();
            if (ref.equals(ud.reference)) {
                item = child;
                found = true;
                break;
            }
        }
        assert forFilter || found;
        return found ? item : null;
    }

    private Item createMainTableItem(final LNTable table, boolean addFolders) {
        Item ret = createTableItem(table, null, false, addFolders);
        ((TTVUserData) ret.getUserData()).type = ItemType.MAINTABLE;

        if (!table.getToReferences().isEmpty() && addFolders) {
            Item toItem = mTree.createItem();
            TTVUserData toUD = new TTVUserData(table);
            toUD.isSubFolder = true;
            toItem.setUserData(toUD);
            toItem.setText(new String[] { LABELS.getRefersTo().getLabel() }, null, null);
            toItem.addItem(createSpinnyThingy());

            ret.addItem(toItem);
        }

        if (!table.getFromReferences().isEmpty() && addFolders) {
            Item fromItem = mTree.createItem();
            TTVUserData fromUD = new TTVUserData(table);
            fromUD.isSubFolder = true;
            fromUD.isFromReference = true;
            fromItem.setUserData(fromUD);
            fromItem.setText(new String[] { LABELS.getReferredBy().getLabel() }, null, null);
            fromItem.addItem(createSpinnyThingy());

            ret.addItem(fromItem);
        }

        return ret;
    }

    private Item createTableItem(LNReference ref, boolean addLoadingIndicator) {
        return createTableItem(ref.isFromReference() ? ref.getFromTable() : ref.getToTable(),
                ref, ref.isFromReference(), addLoadingIndicator);
    }

    private Item createTableItem(final LNTable table, LNReference ref, boolean fromRef, boolean addFolders) {
        Item ret = mTree.createItem();
        if (ref == null) {
            ret.setUserData(new TTVUserData(table));
            ret.setText(new String[] { table.getDescription() +
                    WBRtlUtils.forceLeftToRight(" (" + table.getCode() + ")") }, null, null);
        } else {
            ret.setText(new String[] {
                    MESSAGES.getReferenceViaReference(table.getDescription(),
                            table.getCode(),
                            ref.getFromField().getDescription(),
                            ref.getFromTable().getCode() + "." + ref.getFromField().getCode())
                    }, null, null);

            if (fromRef) {
                ret.setUserData(new TTVUserData(table, ref));
                if (addFolders) {
                    ret.add(createReferenceFolder(table, false, true));
                    ret.add(createReferenceFolder(table, true, true));
                }
            } else {
                HandlerRegistration reg = null;
                if (addFolders) {
                    reg = ret.addStateChangeHandler(tablestatechangehandler);
                    ret.addItem(createSpinnyThingy());
                }
                ret.setUserData(new TTVUserData(table, ref, reg));
            }
        }
        ret.addClickHandler(mTableClickHandler);
        return ret;
    }

    private Item createReferenceFolder(LNTable table, boolean refFrom, boolean addLoadingIndicator) {
        Item folder = mTree.createItem();
        folder.setText(new String[] {
                    refFrom ? LABELS.getReferredBy().getLabel() : LABELS.getRefersTo().getLabel()
                }, null, null);

        HandlerRegistration reg = addLoadingIndicator ? folder.addStateChangeHandler(tablestatechangehandler) : null;
        folder.setUserData(new TTVUserData(refFrom, table, reg));

        if (addLoadingIndicator) {
            folder.addItem(createSpinnyThingy());
        }

        return folder;
    }

    private Item createSpinnyThingy() {
        // Add the spinny thingy so that the folder can be expanded by the user
        Item spinnythingy = mTree.createItem();
        spinnythingy.setSvgIcon("icon-in-progress");
        spinnythingy.setText(new String[] { LABELS.getLoadingReferences().getLabel() }, null, null);
        return spinnythingy;
    }

    @Override
    public void changeFilterStatus(boolean hasFilter, boolean filterActive) {
        //update the cancel / reset button
        String cancelIconId = "";
        if (filterActive) {
            cancelIconId = "icon-cancel";
        } else if (hasFilter) {
            cancelIconId = "icon-reset";
        }
        mStopFilterButton.setSvgIcon(new SvgIcon(cancelIconId));
        mStopFilterButton.setVisible(!"".equals(cancelIconId));

        //update the loading indicator on the field
        if (filterActive) {
            mLoading.start(mFilterField.getElement());
        } else {
            mLoading.stop();
        }

        //update the search button
        String iconId = filterActive ? "" : "icon-search";
        mFilterField.setSvgIcon(new SvgIcon(iconId));
        mFilterField.setEnabled(!filterActive);

        if (!hasFilter) {
            mFilterInputField.setValue("", false);
        }
    }


    /**
     * Specific in-field Loading indicator while searching / rendering.
     * Shamelessly stolen from LN UI's SideNavigation class.
     */
    private static class Loading extends LoadingIndicator {
        Loading() {
            super(false);
            setTransparentStyle(true);
            addStyleName("Loading");
        }

        @Override
        public synchronized void start(final Element reference) {
            // adjust the popup position in the field.
            final PopupPanel panel = getPopupPanel();
            panel.setPopupPositionAndShow(new PositionCallback() {
                @Override
                public void setPosition(int offsetWidth, int offsetHeight) {
                    int padding = 5;
                    int left = LocaleInfo.getCurrentLocale().isRTL()
                            ? reference.getAbsoluteRight() - offsetWidth - padding
                            : reference.getAbsoluteLeft() + padding;

                    int top = reference.getAbsoluteTop() + reference.getOffsetHeight() / 2 - offsetHeight / 2;
                    panel.setPopupPosition(left, top);
                }
            });
        }

        @Override
        public synchronized void stop() {
            if (isVisible()) {
                getPopupPanel().hide();
            }
        }
    }

    private enum ItemType {
        /** Main table item. */
        MAINTABLE,
        /** Subfolder of main table (has no reference). */
        MAINSUB,
        /** Table with subfolders. */
        TABLEFOLDER,
        /** Sub-folder of table. */
        SUBFOLDER,
        /** Table without sub-folders. */
        TABLE
    }

    private static class SortedReferenceBag implements Comparator<LNReference>, Iterable<LNReference> {
        private boolean mIsReset = false;
        private TreeSet<LNReference> mSet;
        private SortMode mSortMode;
        private boolean mUseFromRef;

        SortedReferenceBag() {
            mSet = new TreeSet<>(this);
        }

        public void reset(SortMode sortmode, boolean useFromRef) {
            mSet.clear();
            mSortMode = sortmode;
            mUseFromRef = useFromRef;
            mIsReset = true;
        }

        void add(LNReference ref) {
            assert !mSet.isEmpty() || mIsReset : "SortedReferenceBag must be reset before adding new references";
            assert ref != null : "Cannot add null references to this map";
            mSet.add(ref);
            mIsReset = false;
        }

        @Override
        public int compare(LNReference o1, LNReference o2) {
            // Sort by table (code/desc) (desc/code), then field (code/desc) (desc/code)
            // where code/desc switches depending on sort mode
            LNTable t1 = mUseFromRef ? o1.getFromTable() : o1.getToTable();
            LNTable.Field f1 = o1.getFromField();
            LNTable t2 = mUseFromRef ? o2.getFromTable() : o2.getToTable();
            LNTable.Field f2 = o2.getFromField();

            String[] s1 = new String[4];
            String[] s2 = new String[4];
            switch (mSortMode) {
            case BY_DESC:
                s1[0] = t1.getDescription();
                s1[1] = t1.getCode();
                s1[2] = f1.getDescription();
                s1[3] = f1.getCode();
                s2[0] = t2.getDescription();
                s2[1] = t2.getCode();
                s2[2] = f2.getDescription();
                s2[3] = f2.getCode();
                break;
            case BY_NAME:
                s1[0] = t1.getCode();
                s1[1] = t1.getDescription();
                s1[2] = f1.getCode();
                s1[3] = f1.getDescription();
                s2[0] = t2.getCode();
                s2[1] = t2.getDescription();
                s2[2] = f2.getCode();
                s2[3] = f2.getDescription();
                break;
            default:
                assert false;
                return 0;
            }
            for (int i = 0; i < s1.length; ++i) {
                int c = Integer.signum(s1[i].compareTo(s2[i]));
                switch (c) {
                case -1:
                case 1:
                    return c;
                case 0:
                default:
                    break;
                }
            }
            return 0;
        }

        @Override
        public Iterator<LNReference> iterator() {
            return Collections.unmodifiableSortedSet(mSet).iterator();
        }
    }

    private static class TTVUserData {
        ItemType type;
        /** Indicates whether this item is a sub-folder ('refers to' or 'referred by'). */
        boolean isSubFolder;
        /** Indicates whether this item is a 'referred by' folder. */
        boolean isFromReference;
        /** The {@link LNTable} this item belongs to. */
        LNTable table;
        /** The {@link LNReference} from the parent table to this item's table. Can be null. */
        LNReference reference;
        /** The {@link HandlerRegistration} for lazy-loading items. Can be null. */
        HandlerRegistration registration;

        /**
         * Instantiate a {@link TTVUserData} for a sub-folder of the main table. These are special in that they
         * do not have a reference.
         * @param table the {@link LNTable} that is the main table.
         */
        TTVUserData(LNTable table) {
            this(table, null, null);
            this.type = ItemType.MAINSUB;
        }

        /**
         * Instantiate a {@link TTVUserData} for a table that contains sub-folders.
         * @param table the {@link LNTable} this item belongs to.
         * @param reference the {@link LNReference} from the parent table to this item's table.
         */
        TTVUserData(LNTable table, LNReference reference) {
            this(table, reference, null);
            this.type = ItemType.TABLEFOLDER;
        }

        /**
         * Instantiate a {@link TTVUserData} for a table that does not contain sub-folders.
         * @param table the {@link LNTable} this item belongs to.
         * @param reference the {@link LNReference} from the parent table to this item's table.
         * @param registration the {@link LNReference} from the parent table to this item's table.
         */
        TTVUserData(LNTable table, LNReference reference, HandlerRegistration registration) {
            this.isSubFolder = false;
            this.table = table;
            this.reference = reference;
            this.registration = registration;
            this.type = ItemType.TABLE;
        }

        /**
         * Instantiate a {@link TTVUserData} for a sub-folder ('refers to' or 'referred by').
         * @param isFromReference indicates whether the folder is the referred-by folder.
         * @param table the LNTable to which the item belongs.
         * @param registration the {@link HandlerRegistration} of the item.
         */
        TTVUserData(boolean isFromReference, LNTable table, HandlerRegistration registration) {
            this.isSubFolder = true;
            this.isFromReference = isFromReference;
            this.table = table;
            this.registration = registration;
            this.type = ItemType.SUBFOLDER;
        }
    }
}
