/**
 * Begin Copyright Notice
 *
 * NOTICE
 *
 * THIS SOFTWARE IS THE PROPERTY OF AND CONTAINS CONFIDENTIAL INFORMATION OF
 * INFOR AND/OR ITS AFFILIATES OR SUBSIDIARIES AND SHALL NOT BE DISCLOSED
 * WITHOUT PRIOR WRITTEN PERMISSION. LICENSED CUSTOMERS MAY COPY AND ADAPT
 * THIS SOFTWARE FOR THEIR OWN USE IN ACCORDANCE WITH THE TERMS OF THEIR
 * SOFTWARE LICENSE AGREEMENT. ALL OTHER RIGHTS RESERVED.
 *
 * (c) COPYRIGHT 2018 INFOR. ALL RIGHTS RESERVED. THE WORD AND DESIGN MARKS
 * SET FORTH HEREIN ARE TRADEMARKS AND/OR REGISTERED TRADEMARKS OF INFOR
 * AND/OR ITS AFFILIATES AND SUBSIDIARIES. ALL RIGHTS RESERVED. ALL OTHER
 * TRADEMARKS LISTED HEREIN ARE THE PROPERTY OF THEIR RESPECTIVE OWNERS.
 *
 * End Copyright Notice
 */
package com.infor.ln.wb.common.client.presenter;

import java.util.List;

import com.google.web.bindery.event.shared.HandlerRegistration;
import com.infor.ln.wb.common.client.events.TableSelectionChangeEvent;
import com.infor.ln.wb.common.shared.model.LNReference;
import com.infor.ln.wb.common.shared.model.LNReference.ReferencePath;
import com.infor.ln.wb.common.shared.model.LNTable;

/**
 * This is the interface required for views that show a tree of {@link LNTable}s.
 * <br><br>
 * The view can be filled using {@link #addMainTable} and {@link #addTables}. The view can also directly request an
 * update of the tree using its reference to the {@link ITableTreePresenter}.
 * <br>
 * The view can jump to a specific table using {@link #selectTable}.
 * <br>
 * Filtering works by hiding all items, and then adding specific items that match the filter (and their parent). See the
 * methods {@link #addMainTableForFilter} and {@link #addTableForFilter}.
 */
public interface ITableTreeView {
    /**
     * Set the {@link ITableTreePresenter} for the view to use.
     * @param presenter an {@link ITableTreePresenter}.
     */
    void setPresenter(ITableTreePresenter presenter);

    /**
     * Remove all items from the view.
     */
    void clear();

    /**
     * Add the main table to the view.
     * @param table the {@link LNTable} at the root of the tree.
     */
    void addMainTable(LNTable table);

    /**
     * Add a set of tables to the view at once. These must be all from-references, or all to-references, to/from the
     * same table.
     * @param commonPath the {@link ReferencePath} from the main table to the common parent of all references.
     * @param fromRefs indicates whether the references are tables referring to the table (true) or referred by
     * the table (false).
     * @param tables the {@link List} of {@link LNReference}s to add. Each reference must have either the same to-table,
     * or the same from-table as per the <code>fromRefs</code> parameter.
     */
    void addTables(ReferencePath commonPath, boolean fromRefs, List<LNReference> tables);

    /**
     * Add the main table to the view while filtering. The view should not automatically add sub-folders. The presenter
     * should not assume the view automatically clears the tree when calling this function.
     * @param table the {@link LNTable} at the root of the tree.
     */
    void addMainTableForFilter(LNTable table);

    /**
     * Add a specific table while filtering. The view should add only the table item, no sub-folders.
     * @param tablePath the {@link ReferencePath} from the main table to the target table.
     * @param tableRef the {@link LNReference} from the last entry in the path to the table to add.
     */
    void addTableForFilter(ReferencePath tablePath, LNReference tableRef);

    /**
     * Update the status of the filter field. The filter can be inactive (the initial state), processing, or done. When
     * processing, the view should offer a way for the user to interrupt the process. When done, the view should offer a
     * way to reset the tree.
     * @param hasFilter if true, a filter is present.
     * @param filterActive  if true, a filter is in process.
     */
    void changeFilterStatus(boolean hasFilter, boolean filterActive);

    /**
     * Select a particular table in the view.
     * @param table the {@link LNTable} to select.
     * @param tablePath the {@link ReferencePath} from the main table to the target table.
     */
    void selectTable(LNTable table, ReferencePath tablePath);

    /**
     * Add a handler for {@link TableSelectionChangeEvent}s. The view should fire this event when the user clicks on a
     * table in the tree.
     * @param handler the {@link TableSelectionChangeEvent} handler that should be called.
     * @return a {@link HandlerRegistration} using which the handler can be removed.
     */
    HandlerRegistration addSelectionChangeHandler(TableSelectionChangeEvent.Handler handler);
}
