/**
 * Begin Copyright Notice
 *
 * NOTICE
 *
 * THIS SOFTWARE IS THE PROPERTY OF AND CONTAINS CONFIDENTIAL INFORMATION OF
 * INFOR AND/OR ITS AFFILIATES OR SUBSIDIARIES AND SHALL NOT BE DISCLOSED
 * WITHOUT PRIOR WRITTEN PERMISSION. LICENSED CUSTOMERS MAY COPY AND ADAPT
 * THIS SOFTWARE FOR THEIR OWN USE IN ACCORDANCE WITH THE TERMS OF THEIR
 * SOFTWARE LICENSE AGREEMENT. ALL OTHER RIGHTS RESERVED.
 *
 * (c) COPYRIGHT 2017 INFOR. ALL RIGHTS RESERVED. THE WORD AND DESIGN MARKS
 * SET FORTH HEREIN ARE TRADEMARKS AND/OR REGISTERED TRADEMARKS OF INFOR
 * AND/OR ITS AFFILIATES AND SUBSIDIARIES. ALL RIGHTS RESERVED. ALL OTHER
 * TRADEMARKS LISTED HEREIN ARE THE PROPERTY OF THEIR RESPECTIVE OWNERS.
 *
 * End Copyright Notice
 */
package com.infor.ln.wb.common.shared.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import com.google.gwt.user.client.rpc.IsSerializable;


/**
 * Description of a table in LN, including its fields and references.
 *
 * The JAXB bindings expect the following structure (where the nested <code>field</code> tags are {@link Field}s):
 * <pre>
 * &lt;table code={code} description={description}&gt;
 *   (&lt;field /&gt;)*
 * &lt;/table&gt;
 * </pre>
 */
@XmlRootElement(name = "table")
public class LNTable implements IsSerializable {
    @XmlAttribute(name = "code")
    private String mCode;

    @XmlAttribute(name = "description")
    private String mDescription;

    @XmlElement(name = "field")
    private List<Field> mFields = new ArrayList<>();

    @XmlTransient
    private List<LNReference> mToReferences = new ArrayList<>();

    @XmlTransient
    private List<LNReference> mFromReferences = new ArrayList<>();

    @SuppressWarnings("unused") // Used by GWT
    private LNTable() {
    }

    /**
     * Instantiate a new {@link LNTable}.
     * @param name the name of the table.
     * @param description the description of the table.
     * @param fields a {@link List} with {@link Field}s of the table.
     */
    public LNTable(String name, String description, List<Field> fields) {
        mCode = name;
        mDescription = description;
        mFields.addAll(fields);
    }

    /**
     * @return the table code.
     */
    public String getCode() {
        return mCode;
    }

    /**
     * @return the description of the table.
     */
    public String getDescription() {
        return mDescription;
    }

    /**
     * @return a read-only view on the fields of the table.
     */
    public List<Field> getFields() {
        return Collections.unmodifiableList(mFields);
    }

    /**
     * Find the {@link Field} with the given name in the table.
     * @param name the name of the field (without table name).
     * @param element the element number of the field.
     * @return the {@link Field} with the given name, or null if it is not present in this table.
     */
    public Field findField(String name, int element) {
        for (Field f : mFields) {
            if (name.equals(f.getCode()) && element == f.getElement()) {
                return f;
            }
        }
        return null;
    }

    /**
     * Add all fields from the other table to this table, if not present.
     * @param other the {@link LNTable} whose fields should be added.
     */
    public void addFields(LNTable other) {
        assert mCode.equals(other.mCode) : "Can only add fields from the same table";
        for (Field of : other.mFields) {
            boolean present = false;
            for (Field tf : this.mFields) {
                if (tf.equals(of)) {
                    present = true;
                    break;
                }
            }
            if (!present) {
                mFields.add(of);
            }
        }
    }

    /**
     * @return a read-only view on the references from this table to other tables.
     */
    public List<LNReference> getToReferences() {
        return Collections.unmodifiableList(mToReferences);
    }

    /**
     * Add all given references to this table (if not present yet).
     * @param references the list of {@link LNReference} to add.
     */
    public void addToReferences(List<LNReference> references) {
        for (LNReference ref : references) {
            assert !ref.isFromReference() : "list should only contain to-references";
            addReference(ref);
        }
    }

    /**
     * @return a read-only view on the references from other tables to this table.
     */
    public List<LNReference> getFromReferences() {
        return Collections.unmodifiableList(mFromReferences);
    }

    /**
     * Add a reference to this table; the reference can be either a from- or to-reference.
     * @param ref the {@link LNReference} to add.
     */
    public void addReference(LNReference ref) {
        assert ref.isFromReference() ? ref.getToTable() == this : ref.getFromTable() == this
                : "Reference does not belong to this table";
        List<LNReference> list = ref.isFromReference() ? mFromReferences : mToReferences;
        for (LNReference curref : list) {
            if (curref.equals(ref)) {
                return;
            }
        }
        list.add(ref);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((mCode == null) ? 0 : mCode.hashCode());
        result = prime * result + ((mDescription == null) ? 0 : mDescription.hashCode());
        result = prime * result + ((mFields == null) ? 0 : mFields.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof LNTable)) {
            return false;
        }
        LNTable other = (LNTable) obj;
        return Objects.equals(mCode, other.mCode) &&
                Objects.equals(mDescription, other.mDescription) &&
                Objects.equals(mFields, other.mFields);
    }

    @Override
    public String toString() {
        return "LNTable(" + mCode + ")";
    }


    /**
     * A field in an {@link LNTable}.
     *
     * The JAXB bindings expect the following structure:
     * <pre>
     * &lt;field   number={sequence number}
     *          code={code}
     *          element={element}
     *          description={description}
     *          refersTo={table code}
     *          printable=(true|false) /&gt;
     * </pre>
     */
    public static class Field implements IsSerializable {
        @XmlAttribute(name = "number")
        private int mNumber;

        @XmlAttribute(name = "code")
        private String mCode;

        @XmlAttribute(name = "element")
        private byte mElement;

        @XmlAttribute(name = "description")
        private String mDescription;

        @XmlAttribute(name = "refersTo")
        private String mRefersToTableName;

        @XmlAttribute(name = "printable")
        private boolean mPrintable;

        @XmlTransient
        private LNTable mRefersToTable;

        @SuppressWarnings("unused") // Used by GWT
        private Field() {
        }

        /**
         * Instantiate a new {@link Field}.
         * @param name the name of the field.
         * @param description the description of the field.
         * @param refersToTable the name of the table the field refers to.
         */
        public Field(String name, String description, String refersToTable) {
            mCode = name;
            mDescription = description;
            mRefersToTableName = refersToTable;
        }

        /**
         * @return the sequence number of the field.
         */
        public int getNumber() {
            return mNumber;
        }

        /**
         * @return the code of the field.
         */
        public String getCode() {
            return mCode;
        }

        /**
         * @return the element number of the field.
         */
        public byte getElement() {
            return mElement;
        }

        /**
         * @return the description of the field.
         */
        public String getDescription() {
            return mDescription;
        }

        /**
         * @return the name of the table this field refers to.
         */
        public String getRefersToTableName() {
            return mRefersToTableName;
        }

        /**
         * @return the {@link LNTable} this field returns to.
         */
        public LNTable getRefersToTable() {
            return mRefersToTable;
        }

        /**
         * Update the {@link LNTable} corresponding to the table this field refers to. Must be a table with the same
         * code as {@link #getRefersToTableName()} returns.
         * @param refersToTable the {@link LNTable} this field refers to.
         */
        public void setRefersToTable(LNTable refersToTable) {
            assert refersToTable == null || refersToTable.getCode().equals(mRefersToTableName)
                    : "Name of table does not match";
            mRefersToTable = refersToTable;
        }

        /**
         * @return whether this field can be printed on reports. This not the case for e.g. combined fields and blobs.
         */
        public boolean isPrintable() {
            return mPrintable;
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + ((mCode == null) ? 0 : mCode.hashCode());
            result = prime * result + ((mDescription == null) ? 0 : mDescription.hashCode());
            result = prime * result + ((mRefersToTableName == null) ? 0 : mRefersToTableName.hashCode());
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof Field)) {
                return false;
            }
            Field other = (Field) obj;
            return mNumber == other.mNumber &&
                    mElement == other.mElement &&
                    Objects.equals(mCode, other.mCode) &&
                    Objects.equals(mDescription, other.mDescription) &&
                    Objects.equals(mRefersToTableName, other.mRefersToTableName);
        }
    }

    /**
     * {@link List} of {@link LNTable} with JAXB annotations. Expected structure:
     * <pre>
     * &lt;tables&gt;
     *   (&lt;table /&gt;)*
     * &lt;/tables&gt;
     * </pre>
     */
    @XmlRootElement(name = "tables")
    public static class LNTableList implements IsSerializable {
        /** The list of tables. */
        @XmlElement(name = "table")
        public List<LNTable> tables = new ArrayList<>();
    }
}
