/**
 * Begin Copyright Notice
 *
 * NOTICE
 *
 * THIS SOFTWARE IS THE PROPERTY OF AND CONTAINS CONFIDENTIAL INFORMATION OF
 * INFOR AND/OR ITS AFFILIATES OR SUBSIDIARIES AND SHALL NOT BE DISCLOSED
 * WITHOUT PRIOR WRITTEN PERMISSION. LICENSED CUSTOMERS MAY COPY AND ADAPT
 * THIS SOFTWARE FOR THEIR OWN USE IN ACCORDANCE WITH THE TERMS OF THEIR
 * SOFTWARE LICENSE AGREEMENT. ALL OTHER RIGHTS RESERVED.
 *
 * (c) COPYRIGHT 2018 INFOR. ALL RIGHTS RESERVED. THE WORD AND DESIGN MARKS
 * SET FORTH HEREIN ARE TRADEMARKS AND/OR REGISTERED TRADEMARKS OF INFOR
 * AND/OR ITS AFFILIATES AND SUBSIDIARIES. ALL RIGHTS RESERVED. ALL OTHER
 * TRADEMARKS LISTED HEREIN ARE THE PROPERTY OF THEIR RESPECTIVE OWNERS.
 *
 * End Copyright Notice
 */
package com.infor.ln.wb.common.server.rpc;

import org.owasp.esapi.ESAPI;
import org.owasp.esapi.Logger;

import com.infor.erpln.jca.dllclient.BaanDLLCallException;
import com.infor.ln.wb.common.server.RequestInput;
import com.infor.ln.wb.common.shared.BMLException;
import com.infor.ln.wb.common.shared.model.LNTable;
import com.infor.ln.wb.common.shared.model.LNTable.LNTableList;
import com.infor.ln.wb.common.shared.resources.CommonLNEnums;
import com.infor.ln.wb.common.shared.resources.CommonLNLabels;
import com.infor.ln.wb.common.shared.resources.CommonLNMessages;
import com.infor.ln.wb.common.shared.rpc.IWbCommonService;
import com.infor.ln.workbench.server.dll.DLLTools;
import com.infor.ln.workbench.server.rpc.BaseService;
import com.infor.ln.workbench.server.web.WorkbenchState;
import com.infor.ln.workbench.shared.resources.LNResources;
import com.infor.ln.workbench.shared.rpc.ServerRPCException;

/**
 *
 */
public class WbCommonService extends BaseService implements IWbCommonService {
    private static final long serialVersionUID = 1L;
    private static final String E_CALL_FAILED = "Backend call failed";
    private static final String E_CALL_RETURNED_FAILURE = "Backend call returned failure";
    private static final String E_CALL_NO_OBJECT = "Backend call failed to return an object";
    private static final String BACKEND_LIBRARY = "ottextdllwbcomm";
    private static final String BACKEND_FUNCTION = "RequestHandler.Execute";

    private static final Logger LOG = ESAPI.getLogger(WbCommonService.class);

    @Override
    public LNResources<CommonLNLabels, CommonLNEnums, CommonLNMessages> getResources(String connectionId)
            throws ServerRPCException {
        DLLTools dllTools = new DLLTools(getWorkbenchState(connectionId));

        CommonLNLabels labels = new CommonLNLabels();
        CommonLNEnums enums = new CommonLNEnums();
        CommonLNMessages messages = new CommonLNMessages();
        LNResources<CommonLNLabels, CommonLNEnums, CommonLNMessages> resources = new LNResources<>(
                labels, enums, messages);

        try {
            return dllTools.executeXmlMethod1("ottdllwb.label", "get.labels", resources,
                    CommonLNLabels.class, CommonLNEnums.class, CommonLNMessages.class);
        } catch (BaanDLLCallException e) {
            LOG.error(Logger.EVENT_FAILURE, E_CALL_FAILED, e);
            throw convertToServerRPCException(e);
        }
    }

    @Override
    public LNTableList getTableReferences(String connectionId, String tableName, String dir)
            throws ServerRPCException {
        RequestInput request = new RequestInput("getTableReferences");
        request.addParam("table", tableName);
        request.addParam("direction", dir);

        RequestOutput result = tryBackendCall(getWorkbenchState(connectionId), request);
        LNTableList tables = tryConvertObject(LNTableList.class, result);
        LOG.info(Logger.EVENT_UNSPECIFIED, "number of tables: " + tables.tables.size());
        return tables;
    }

    @Override
    public LNTable getTableFields(String connectionId, String tableName) throws ServerRPCException {
        RequestInput request = new RequestInput("getTableFields");
        request.addParam("table", tableName);

        RequestOutput result = tryBackendCall(getWorkbenchState(connectionId), request);
        return result.convertObject(LNTable.class);
    }

    private RequestOutput tryBackendCall(WorkbenchState state, RequestInput request) throws ServerRPCException {
        final DLLTools dllTools = new DLLTools(state);
        RequestOutput result = new RequestOutput();
        try {
            result = dllTools.executeXmlMethod2(BACKEND_LIBRARY, BACKEND_FUNCTION, request, result);
        } catch (Exception e) {
            LOG.error(Logger.EVENT_FAILURE, E_CALL_FAILED, e);
            throw convertToServerRPCException(e);
        }
        for (final String error : result.getErrors()) {
            LOG.warning(Logger.EVENT_UNSPECIFIED, error);
        }
        if (!result.hasSuccess()) {
            Exception e = new BMLException(result.getErrors().get(0));
            LOG.error(Logger.EVENT_FAILURE, E_CALL_RETURNED_FAILURE, e);
            throw convertToServerRPCException(e);
        }
        return result;
    }

    private <T> T tryConvertObject(Class<T> clazz, RequestOutput result) throws ServerRPCException {
        T obj = result.convertObject(clazz);
        if (obj == null) {
            Exception e = new BMLException(E_CALL_NO_OBJECT);
            LOG.error(Logger.EVENT_FAILURE, "No object returned by backend", e);
            throw convertToServerRPCException(e);
        }
        return obj;
    }


}
