/**
 * Begin Copyright Notice
 *
 * NOTICE
 *
 * THIS SOFTWARE IS THE PROPERTY OF AND CONTAINS CONFIDENTIAL INFORMATION OF
 * INFOR AND/OR ITS AFFILIATES OR SUBSIDIARIES AND SHALL NOT BE DISCLOSED
 * WITHOUT PRIOR WRITTEN PERMISSION. LICENSED CUSTOMERS MAY COPY AND ADAPT
 * THIS SOFTWARE FOR THEIR OWN USE IN ACCORDANCE WITH THE TERMS OF THEIR
 * SOFTWARE LICENSE AGREEMENT. ALL OTHER RIGHTS RESERVED.
 *
 * (c) COPYRIGHT 2016 INFOR. ALL RIGHTS RESERVED. THE WORD AND DESIGN MARKS
 * SET FORTH HEREIN ARE TRADEMARKS AND/OR REGISTERED TRADEMARKS OF INFOR
 * AND/OR ITS AFFILIATES AND SUBSIDIARIES. ALL RIGHTS RESERVED. ALL OTHER
 * TRADEMARKS LISTED HEREIN ARE THE PROPERTY OF THEIR RESPECTIVE OWNERS.
 *
 * End Copyright Notice
 */

package com.infor.ln.wb.common.server.rpc;

import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlAnyElement;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.w3c.dom.Node;

import com.infor.ln.wb.common.shared.ResponseStatus;
import com.infor.ln.workbench.server.JAXBUtils;

/**
 * Describes the output of an RPC request. Optionally contains an object and/or errors.
 */
@XmlRootElement(name = "result")
public class RequestOutput {
    @XmlAttribute
    private String action;
    @XmlAttribute
    private ResponseStatus status;
    @XmlElement(name = "error")
    private final List<String> errors = new ArrayList<>();

    // Node is an interface, so JAXB cannot instantiate that directly
    // It is stored as an Object by JAXB, then casted afterwards (cf. afterUnmarshal)
    // It is annotated with XmlAnyElement because some classes are used as top-level objects, as
    // well as nested objects; e.g. LabelVariantList can be contained in ReportAdditionalInformation.
    @XmlAnyElement
    private List<Object> objects;
    @XmlTransient
    private List<Node> nodes = new ArrayList<>();

    /**
     * Indicates whether the call was successful (according to the backend).
     * @return true if the backend reported success.
     */
    public boolean hasSuccess() {
        return this.status == ResponseStatus.OK;
    }

    /**
     * @return true if errors are present.
     * Note that errors can be present for a successful call.
     */
    public boolean hasErrors() {
        return !this.errors.isEmpty();
    }
    /**
     * @return a copy of all error messages.
     */
    public List<String> getErrors() {
        return new ArrayList<>(this.errors);
    }

    /**
     * Converts the contained object to another class using its JAXB bindings.
     * @param <T> the target class to be returned.
     * @param clazz the target class containing the bindings (should be {@code <T>.class}).
     * @return the contained object converted to {@code clazz}, or null if no object is present.
     */
    public <T> T convertObject(final Class<T> clazz) {
        if (this.nodes.isEmpty()) {
            return null;
        }
        return JAXBUtils.createObjectFromNode(this.nodes.get(0), clazz);
    }

    /**
     * Converts the contained list of objects to another class using its JAXB bindings.
     * @param <T> the target class to be contained in the list.
     * @param clazz the target class containing the bindings (should be {@code <T>.class}).
     * @return a (possibly empty) {@link List} containing objects of the specified type.
     */
    public <T> List<T> convertList(final Class<T> clazz) {
        List<T> ret = new ArrayList<T>();
        for (Node n : this.nodes) {
            ret.add(JAXBUtils.createObjectFromNode(n, clazz));
        }
        return ret;
    }

    /**
     * Get the text content of the object node.
     * @return the text content of the node, or null if there is no object.
     */
    public String getObjectText() {
        return nodes.isEmpty() ? null : nodes.get(0).getTextContent();
    }

    @SuppressWarnings("unused") // called by JAXB
    private void afterUnmarshal(final Unmarshaller unmarshaller, final Object parent) {
        assert this.status != null : "Invalid status value in response";
        if (this.objects != null) {
            for (Object o : this.objects) {
                assert o instanceof Node;
                this.nodes.add((Node) o);
            }
        }
    }
}
