/**
 * Begin Copyright Notice
 *
 * NOTICE
 *
 * THIS SOFTWARE IS THE PROPERTY OF AND CONTAINS CONFIDENTIAL INFORMATION OF
 * INFOR AND/OR ITS AFFILIATES OR SUBSIDIARIES AND SHALL NOT BE DISCLOSED
 * WITHOUT PRIOR WRITTEN PERMISSION. LICENSED CUSTOMERS MAY COPY AND ADAPT
 * THIS SOFTWARE FOR THEIR OWN USE IN ACCORDANCE WITH THE TERMS OF THEIR
 * SOFTWARE LICENSE AGREEMENT. ALL OTHER RIGHTS RESERVED.
 *
 * (c) COPYRIGHT 2017 INFOR. ALL RIGHTS RESERVED. THE WORD AND DESIGN MARKS
 * SET FORTH HEREIN ARE TRADEMARKS AND/OR REGISTERED TRADEMARKS OF INFOR
 * AND/OR ITS AFFILIATES AND SUBSIDIARIES. ALL RIGHTS RESERVED. ALL OTHER
 * TRADEMARKS LISTED HEREIN ARE THE PROPERTY OF THEIR RESPECTIVE OWNERS.
 *
 * End Copyright Notice
 */
package com.infor.ln.wb.common.client.view;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.google.gwt.json.client.JSONNumber;
import com.google.gwt.json.client.JSONObject;
import com.google.gwt.json.client.JSONParser;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.ResizeComposite;
import com.google.gwt.view.client.SingleSelectionModel;
import com.google.web.bindery.event.shared.EventBus;
import com.google.web.bindery.event.shared.HandlerRegistration;
import com.google.web.bindery.event.shared.SimpleEventBus;
import com.infor.component.client.widgets.Label;
import com.infor.component.client.widgets.grid.DataGrid;
import com.infor.component.client.widgets.grid.DataGrid.Personalization;
import com.infor.component.client.widgets.grid.DataGrid.PersonalizationCallback;
import com.infor.component.client.widgets.grid.DefaultSelectionManager;
import com.infor.component.client.widgets.grid.EditableValueColumn;
import com.infor.component.client.widgets.grid.FilterOperatorLibrary;
import com.infor.component.client.widgets.grid.column.BooleanColumn;
import com.infor.component.client.widgets.grid.column.TextColumn;
import com.infor.component.client.widgets.menu.AsyncMenu;
import com.infor.component.shared.Pair;
import com.infor.ln.wb.common.client.WbCommon;
import com.infor.ln.wb.common.client.events.TableFieldSelectionChangeEvent;
import com.infor.ln.wb.common.client.presenter.ITableFieldView;
import com.infor.ln.wb.common.shared.model.LNTable.Field;
import com.infor.ln.wb.common.shared.resources.CommonLNLabels;


/**
 * A view showing table fields in a grid, indicating whether they have been selected for a dataset.
 * <br><br>
 * In addition to the view-presenter interface as defined by {@link ITableFieldView}, this view supports
 * persistent grid personalization through {@link #getPersonalization()}, {@link #setPersonalization(String)}, and
 * {@link #applyDefaultColumnSetup()}.
 */
public class TableFieldView extends ResizeComposite implements ITableFieldView {
    private static final CommonLNLabels LABELS = WbCommon.labels();

    private static final String COLNAME_NAME = "name";
    private static final String COLNAME_ELEM = "elem";
    private static final String COLNAME_DESC = "desc";
    private static final String COLNAME_SEL = "sel";

    private static final String COLPERS_INDEX = "index";
    private static final String COLPERS_SIZE = "size";

    private static final int COLWIDTH_NAME = 12 * 17;
    private static final int COLWIDTH_ELEM = 100;
    private static final int COLWIDTH_DESC = 12 * 35;
    private static final int COLWIDTH_SEL = 100;

    private EventBus mEventBus = new SimpleEventBus();

    private DataGrid<Field> mGrid;
    private TextColumn<Field> mNameColumn;
    private TextColumn<Field> mElementColumn;
    private TextColumn<Field> mDescColumn;
    private BooleanColumn<Field> mSelColumn;
    private boolean mShowElements = true;

    private Set<Field> mFieldSelection = new HashSet<>();

    private JSONObject mGridPersData = new JSONObject();

    private PersonalizationCallback mPersonalizationCallback = new PersonalizationCallback() {
        @Override
        public void onColumnMove(int column, int count, int newIndex) {
            String name = mGrid.getColumn(newIndex).getName();
            JSONObject coldata = mGridPersData.containsKey(name) ? mGridPersData.get(name).isObject() : null;
            if (coldata == null) {
                coldata = new JSONObject();
                mGridPersData.put(name, coldata);
            }
            coldata.put(COLPERS_INDEX, new JSONNumber(newIndex));
        }

        @Override
        public void onColumnResize(int column, int size) {
            String name = mGrid.getColumn(column).getName();
            JSONObject coldata = mGridPersData.containsKey(name) ? mGridPersData.get(name).isObject() : null;
            if (coldata == null) {
                coldata = new JSONObject();
                mGridPersData.put(name, coldata);
            }
            coldata.put(COLPERS_SIZE, new JSONNumber(size));
        }

        @Override
        public void onColumnsHidden(int column, int count) {
            // Not allowed
        }

        @Override
        public void onColumnsShown(int column, int count) {
            // Not allowed
        }

        @Override
        public void onFrozenCountChanged(int newCount) {
            // Not allowed
        }
    };

    /**
     * Instantiate a new, empty, {@link TableFieldView}.
     */
    public TableFieldView() {
        this(true);
    }

    /**
     * Instantiate a new, empty, {@link TableFieldView}.
     * @param showElements indicates whether individual elements of array fields should be shown. This adds an
     * additional column to the grid.
     */
    public TableFieldView(boolean showElements) {
        mShowElements = showElements;
        mGrid = new DataGrid<>();
        initWidget(mGrid);
        setupGrid();
    }

    @Override
    public void setReadonly(boolean readonly) {
        mSelColumn.setReadOnly(readonly);
        mGrid.resetColumn(mSelColumn);
    }


    @Override
    public void setData(List<Field> fields, Set<Field> selectedFields) {
        mFieldSelection.clear();
        mFieldSelection.addAll(selectedFields);

        mGrid.clear();

        mGrid.clearFilter();
        mGrid.setColumnFilterOperator(mDescColumn, FilterOperatorLibrary.getInstance().OPERATOR_CONTAINS);

        List<Field> filteredFields = fields;
        if (!mShowElements) {
            filteredFields = new ArrayList<>(fields);
            Iterator<Field> iter = filteredFields.listIterator();
            while (iter.hasNext()) {
                if (iter.next().getElement() > 1) {
                    iter.remove();
                }
            }
        }
        mGrid.setDataRows(filteredFields);
    }

    @Override
    public void selectField(Field field) {
        mGrid.getSelectionManager().setSelected(field, true);
        mGrid.makeObjectVisible(field);
    }

    @Override
    public void setFieldSelected(Field field, boolean selected) {
        if (selected) {
            mFieldSelection.add(field);
        } else {
            mFieldSelection.remove(field);
        }
        mGrid.refresh(field);
    }

    @Override
    public HandlerRegistration addFieldSelectionChangeHandler(TableFieldSelectionChangeEvent.Handler handler) {
        return mEventBus.addHandler(TableFieldSelectionChangeEvent.TYPE, handler);
    }


    /**
     * Initialize the grid according to the given personalization.
     * @param persdata a {@link String} describing the personalization as returned by {@link #getPersonalization()}.
     */
    public void setPersonalization(String persdata) {
        JSONObject pers = JSONParser.parseStrict(persdata).isObject();
        if (pers != null) {
            mGridPersData = pers;

            @SuppressWarnings("unchecked")
            Pair<String, EditableValueColumn<?, ?, ?, ?>>[] columns = new Pair[] {
                    new Pair<>(COLNAME_NAME, mNameColumn),
                    new Pair<>(COLNAME_ELEM, mElementColumn),
                    new Pair<>(COLNAME_DESC, mDescColumn),
                    new Pair<>(COLNAME_SEL, mSelColumn),
            };

            for (Pair<String, EditableValueColumn<?, ?, ?, ?>> col : columns) {
                JSONObject colpers = pers.containsKey(col.getLeft()) ? pers.get(col.getLeft()).isObject() : null;
                if (colpers != null) {
                    if (colpers.containsKey(COLPERS_INDEX)) {
                        mGrid.setColumnIndex(col.getRight(), (int) colpers.get(COLPERS_INDEX).isNumber().doubleValue());
                    }
                    if (colpers.containsKey(COLPERS_SIZE)) {
                        mGrid.setColumnWidth(col.getRight(), (int) colpers.get(COLPERS_SIZE).isNumber().doubleValue());
                    }
                }
            }
        }
    }

    /**
     * @return a {@link String} describing the current grid personalizations.
     */
    public String getPersonalization() {
        return mGridPersData.toString();
    }

    /**
     * Reset the columns to their default position and size.
     */
    public void applyDefaultColumnSetup() {
        mGrid.setColumnIndex(mNameColumn, 1);
        mGrid.setColumnIndex(mElementColumn, 2);
        mGrid.setColumnIndex(mDescColumn, 3);
        mGrid.setColumnIndex(mSelColumn, 4);
        mGrid.setColumnWidth(mNameColumn, COLWIDTH_NAME);
        mGrid.setColumnWidth(mElementColumn, COLWIDTH_ELEM);
        mGrid.setColumnWidth(mDescColumn, COLWIDTH_DESC);
        mGrid.setColumnWidth(mSelColumn, COLWIDTH_SEL);
    }

    private void setupGrid() {
        mGrid.setSize("100%", "100%");
        mGrid.setGridMenu((AsyncMenu) null);
        mGrid.setFilterVisible(true);
        mGrid.setInstantFiltering(true);

        mGrid.setPersonalizationsAllowed(EnumSet.allOf(Personalization.class), false);
        mGrid.setPersonalizationsAllowed(EnumSet.of(Personalization.COLUMN_MOVE, Personalization.COLUMN_RESIZE), true);
        mGrid.setPersonalizationCallback(mPersonalizationCallback);

        final SingleSelectionModel<Field> selectionModel = new SingleSelectionModel<>();
        mGrid.setSelectionManager(new DefaultSelectionManager<>(mGrid, selectionModel));

        mNameColumn = new TextColumn<Field>() {
            @Override
            public String getValue(Field field) {
                return field.getCode();
            }

            @Override
            protected void onValueChanged(Field field, String newValue) {
                assert false;
            }
        };
        mNameColumn.setReadOnly(true);
        mNameColumn.setFilterCaseSensitive(false);
        mNameColumn.setName(COLNAME_NAME);
        mGrid.addColumn(mNameColumn, new Label(LABELS.getField().getLabel()), COLWIDTH_NAME);

        mElementColumn = new TextColumn<Field>() {
            @Override
            public String getValue(Field field) {
                return field.getElement() == 0 ? "" : Integer.toString(field.getElement());
            }

            @Override
            protected void onValueChanged(Field field, String newValue) {
                assert false;
            }
        };
        mElementColumn.setReadOnly(true);
        mElementColumn.setName(COLNAME_ELEM);
        mGrid.addColumn(mElementColumn, new Label(LABELS.getElement().getLabel()), COLWIDTH_ELEM);
        mGrid.setColumnVisible(mElementColumn, mShowElements);

        mDescColumn = new TextColumn<Field>() {
            @Override
            public String getValue(Field field) {
                return field.getDescription();
            }

            @Override
            protected void onValueChanged(Field field, String newValue) {
                assert false;
            }
        };
        mDescColumn.setReadOnly(true);
        mDescColumn.setFilterCaseSensitive(false);
        mDescColumn.setName(COLNAME_DESC);
        mGrid.addColumn(mDescColumn, new Label(LABELS.getDescription().getLabel()), COLWIDTH_DESC);
        mGrid.setColumnFilterOperator(mDescColumn, FilterOperatorLibrary.getInstance().OPERATOR_CONTAINS, false);

        mSelColumn = new BooleanColumn<Field>() {
            @Override
            public Boolean getValue(Field field) {
                return mFieldSelection.contains(field);
            }

            @Override
            protected void onValueChanged(Field field, boolean selected) {
                mEventBus.fireEvent(new TableFieldSelectionChangeEvent(field.getCode(), field.getElement(), selected));
            }
        };
        mSelColumn.setHorizontalAlignment(HasHorizontalAlignment.ALIGN_CENTER);
        mSelColumn.setName(COLNAME_SEL);
        mGrid.addColumn(mSelColumn, new Label(LABELS.getSelected()), COLWIDTH_SEL);
    }
}
