/*
 * Decompiled with CFR 0.152.
 */
package com.infor.erpln.utils.oauth10;

import com.infor.erpln.utils.oauth10.AuthorizationStatus;
import com.infor.erpln.utils.oauth10.BasicRandom;
import com.infor.erpln.utils.oauth10.Clock;
import com.infor.erpln.utils.oauth10.OAuthHeader;
import com.infor.erpln.utils.oauth10.OAuthUtil;
import com.infor.erpln.utils.oauth10.Random;
import com.infor.erpln.utils.oauth10.Rfc5849Parameters;
import com.infor.erpln.utils.oauth10.UsedNonceCache;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.Base64;
import java.util.Date;
import java.util.Locale;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class Authenticator {
    private static final Logger LOG = LogManager.getLogger(Authenticator.class);
    private static final int MAX_CLOCK_SKEW = 900;
    private final UsedNonceCache m_nonceCache = new UsedNonceCache(900);
    private final String m_key;
    private final String m_secret;
    private Clock m_clock;
    private Random m_random;

    public static Authenticator getInstance(String key, String secret) {
        Authenticator authenticator;
        if (key == null) {
            key = "";
        }
        if (key.isEmpty()) {
            LOG.always().log("Creating fake OAuth 1.0 authenticator - signature verification always succeeds");
            authenticator = new Authenticator(key, secret){

                @Override
                public AuthorizationStatus verify(String method, String rawBaseUri, String queryString, String headerValue) {
                    return new AuthorizationStatus();
                }

                @Override
                public String getOAuthHeaderValue(String method, String uriString) {
                    return "";
                }
            };
        } else {
            authenticator = new Authenticator(key, secret);
        }
        return authenticator;
    }

    public AuthorizationStatus verify(String method, String rawBaseUri, String headerValue) {
        return this.verify(method, rawBaseUri, null, headerValue);
    }

    public AuthorizationStatus verify(String method, String rawBaseUri, String queryString, String headerValue) {
        AuthorizationStatus apiStatus = new AuthorizationStatus().badRequest("Missing or unsupported parameters");
        OAuthHeader oauthHeader = OAuthHeader.getInstance(headerValue);
        if (oauthHeader.isValid()) {
            try {
                if (!oauthHeader.getKey().equals(this.m_key)) {
                    LOG.debug("Client credential '{}' does not match configured '{}'", (Object)oauthHeader.getKey(), (Object)this.m_key);
                    return apiStatus.unauthorized("Invalid client credentials");
                }
                if (!this.isValidTimestamp(oauthHeader.getTimestamp())) {
                    LOG.debug("Timestamp is not valid: {}", (Object)new Date(oauthHeader.getTimestamp() * 1000L));
                    return apiStatus.unauthorized("Invalid timestamp");
                }
                if (!this.m_nonceCache.update(oauthHeader.getUsedNonce())) {
                    LOG.debug("Nonce '{}' is not valid", (Object)oauthHeader.getUsedNonce());
                    return apiStatus.unauthorized("Invalid or used nonce");
                }
                String signature = this.calculateSignature(method, rawBaseUri, queryString, oauthHeader);
                if (!signature.equals(oauthHeader.getSignature())) {
                    LOG.debug("Signature '{}' does not match calculated '{}'", (Object)oauthHeader.getSignature(), (Object)signature);
                    return apiStatus.unauthorized("Invalid signature");
                }
                LOG.debug("OAuth signature OK");
                apiStatus.ok();
            }
            catch (UnsupportedEncodingException | GeneralSecurityException e) {
                LOG.error("Exception when calculating OAuth signature:", (Throwable)e);
                apiStatus.badRequest(e.getMessage());
            }
        } else {
            LOG.info("Authorization header is not valid: {}", (Object)headerValue);
        }
        return apiStatus;
    }

    public String getOAuthHeaderValue(String method, String uriString) {
        OAuthHeader oauthHeader = OAuthHeader.getInstance();
        oauthHeader.setKey(this.m_key);
        oauthHeader.setSignatureMethod(HmacType.HmacSHA256.toString());
        oauthHeader.setTimeStamp(this.m_clock.getTime());
        oauthHeader.setNonce(this.m_random.getRandom());
        oauthHeader.setVersion("1.0");
        try {
            URI uri = new URI(uriString);
            StringBuilder baseUri = new StringBuilder();
            String scheme = uri.getScheme();
            baseUri.append(scheme).append("://");
            baseUri.append(uri.getHost());
            int port = uri.getPort();
            if (!(port <= 0 || scheme.equals("http") && port == 80 || scheme.equals("https") && port == 443)) {
                baseUri.append(':').append(port);
            }
            baseUri.append(uri.getPath());
            String signature = this.calculateSignature(method, baseUri.toString(), uri.getRawQuery(), oauthHeader);
            oauthHeader.setSignature(signature);
            return oauthHeader.getHeaderValueAsString();
        }
        catch (URISyntaxException e) {
            LOG.error("Malformed uriString: {}", (Object)uriString, (Object)e);
        }
        catch (UnsupportedEncodingException | GeneralSecurityException e) {
            LOG.error("Exception when generating OAuth signature", (Throwable)e);
        }
        return "";
    }

    private String calculateSignature(String method, String rawBaseUri, String queryString, OAuthHeader oauthHeader) throws UnsupportedEncodingException, GeneralSecurityException {
        Rfc5849Parameters parameters = new Rfc5849Parameters();
        parameters.addFromQueryString(queryString);
        parameters.addFromOAuthHeader(oauthHeader.getFields());
        StringBuilder signatureBase = new StringBuilder();
        signatureBase.append(method.toUpperCase(Locale.getDefault())).append('&');
        signatureBase.append(OAuthUtil.percentEncode(rawBaseUri)).append('&');
        signatureBase.append(OAuthUtil.percentEncode(parameters.getNormalized()));
        LOG.debug("Signature base string: {}", (Object)signatureBase);
        return Signature.calculateRFC2104HMAC(signatureBase.toString(), this.getSigningKey(), oauthHeader.getHmacType());
    }

    private boolean isValidTimestamp(long timestamp) {
        return Math.abs(this.m_clock.getTime() - timestamp) < 900L;
    }

    private String getSigningKey() {
        return OAuthUtil.percentEncode(this.m_secret) + "&";
    }

    private Authenticator(String key, String secret) {
        this.m_key = key;
        this.m_secret = secret;
        this.m_clock = new Clock(){

            @Override
            public long getTime() {
                return System.currentTimeMillis() / 1000L;
            }
        };
        this.m_random = new BasicRandom(16);
    }

    public void setClock(Clock clock) {
        this.m_clock = clock;
    }

    public void setRandom(Random random) {
        this.m_random = random;
    }

    static enum HmacType {
        HmacSHA1("HMAC-SHA1"),
        HmacSHA256("HMAC-SHA256"),
        HmacSHA512("HMAC-SHA512");

        private final String m_method;

        private HmacType(String method) {
            this.m_method = method;
        }

        public String toString() {
            return this.m_method;
        }

        public static HmacType parse(String method) {
            if (HmacType.HmacSHA1.m_method.equalsIgnoreCase(method)) {
                return HmacSHA1;
            }
            if (HmacType.HmacSHA256.m_method.equalsIgnoreCase(method)) {
                return HmacSHA256;
            }
            if (HmacType.HmacSHA512.m_method.equalsIgnoreCase(method)) {
                return HmacSHA512;
            }
            throw new IllegalArgumentException(method);
        }
    }

    private static class Signature {
        private Signature() {
        }

        public static String calculateRFC2104HMAC(String data, String key, HmacType hmacType) throws GeneralSecurityException {
            SecretKeySpec signingKey = new SecretKeySpec(key.getBytes(StandardCharsets.UTF_8), hmacType.name());
            Mac mac = Mac.getInstance(hmacType.name());
            mac.init(signingKey);
            byte[] rawHmac = mac.doFinal(data.getBytes(StandardCharsets.UTF_8));
            return Base64.getEncoder().encodeToString(rawHmac);
        }
    }
}

